/*
 * Copyright (c) 2003 Duncan Barclay <dmlb@dmlb.org>
 * Modifications for FreeBSD 4.8 by Edwin Groothuis <edwin@mavetju.org>
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define XXX 0

#include <sys/cdefs.h>
__FBSDID("$$");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/module.h>

#include <sys/endian.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/malloc.h>
#include <sys/socket.h>
#include <sys/sysctl.h>

#include <net/if.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <net/if_dl.h>
#include <net/if_media.h>

#include <net/bpf.h>

#include <machine/bus.h>
#include <machine/clock.h>
#include <machine/resource.h>
#include <machine/bus_pio.h>
#include <machine/bus_memio.h>
#include <sys/bus.h>
#include <sys/rman.h>

#include <vm/vm.h>
#include <vm/pmap.h>

#if __FreeBSD_version < 500000
#include <pci/pcireg.h>
#include <pci/pcivar.h>
#endif

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>

#if __FreeBSD_version > 500000
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>
#endif

#include <dev/bcm/if_bcmreg.h>
#include <dev/bcm/if_bcmmib.h>

/* "controller miibus0" required.  See GENERIC if you get errors here. */
#include "miibus_if.h"

static struct bcm_type bcm_devs[] = {
	{ BCOM_VENDORID, BCOM_DEVICEID_4401,
		"Broadcom 10/100 Base-T Ethernet" },
	{0, 0, NULL}
};

static int	bcm_probe(device_t);
static int	bcm_attach(device_t);
static int	bcm_detach(device_t);
static void	bcm_shutdown(device_t);

XXX_SOMETODO int	bcm_ifmedia_upd(struct ifnet *);
XXX_SOMETODO void	bcm_ifmedia_sts(struct ifnet *, struct ifmediareq *);
static void	bcm_init(void *);
static void	bcm_intr(void *);
static int	bcm_ioctl(struct ifnet *, u_long, caddr_t);
static void	bcm_start(struct ifnet *);
static void	bcm_stop(struct bcm_softc *);
XXX_SOMETODO void	bcm_tick(void *);
static void	bcm_watchdog(struct ifnet *);

static void	bcm_ring_rx_eof(struct bcm_softc *);
static void	bcm_ring_rx_free(struct bcm_softc *);
static int	bcm_ring_rx_init(struct bcm_softc *);
static int	bcm_ring_rx_newbuf(struct bcm_softc *, int, struct mbuf *);
static void	bcm_ring_rx_map(void *, bus_dma_segment_t *, int, int);
static int	bcm_ring_tx_encap(struct bcm_softc *, struct mbuf *, int *);
static void	bcm_ring_tx_eof(struct bcm_softc *);
static void	bcm_ring_tx_free(struct bcm_softc *);
static void	bcm_ring_tx_init(struct bcm_softc *);
static void	bcm_ring_tx_map(void *, bus_dma_segment_t *, int, int);
static void	bcm_ring_map(void *, bus_dma_segment_t *, int, int);

static int	bcm_miibus_readreg(device_t, int, int);
static int	bcm_miibus_writereg(device_t, int, int, int);
XXX_SOMETODO void	bcm_miibus_statchg(device_t);
static int	bcm_mi_readphy(struct bcm_softc *, u_int32_t);
static void	bcm_mi_writephy(struct bcm_softc *, u_int32_t, u_int32_t);

static void	bcm_chip_reset(struct bcm_softc *);
static void	bcm_chip_reset_core(struct bcm_softc *);
static void	bcm_chip_reset_phy(struct bcm_softc *);
static void	bcm_chip_enable(struct bcm_softc *);
static void	bcm_chip_enable_core(struct bcm_softc *);
static void	bcm_chip_enable_phy(struct bcm_softc *);
static void	bcm_chip_halt(struct bcm_softc *scp);
static void	bcm_chip_halt_core(struct bcm_softc *scp);
static void	bcm_chip_halt_phy(struct bcm_softc *scp);
static void	bcm_chip_get_core(struct bcm_softc *, struct bcm_core *);
static void	bcm_chip_rxfilter(struct bcm_softc *);
static void	bcm_chip_writecam(struct bcm_softc *, u_int8_t *, u_int32_t);
XXX_SOMETODO void	bcm_chip_clear_stats(struct bcm_softc *);
XXX_ALLTODO void	bcm_chip_get_stats(struct bcm_softc *, struct bcm_mib *);

static devclass_t bcm_devclass;

static device_method_t bcm_methods[] = {
	/* Device interface */
	DEVMETHOD(device_probe,		bcm_probe),
	DEVMETHOD(device_attach,	bcm_attach),
	DEVMETHOD(device_detach,	bcm_detach),
	DEVMETHOD(device_shutdown,	bcm_shutdown),

	/* bus interface */
	DEVMETHOD(bus_print_child,	bus_generic_print_child),
	DEVMETHOD(bus_driver_added,	bus_generic_driver_added),

	/* MII interface */
	DEVMETHOD(miibus_readreg,	bcm_miibus_readreg),
	DEVMETHOD(miibus_writereg,	bcm_miibus_writereg),
	DEVMETHOD(miibus_statchg,	bcm_miibus_statchg),
	{ 0, 0 }
};

static driver_t bcm_driver = {
	"bcm",
	bcm_methods,
	sizeof(struct bcm_softc)
};

#ifdef __i386__
static int bcm_rx_quick = 1;
SYSCTL_INT(_hw, OID_AUTO, bcm_rx_quick, CTLFLAG_RW,
        &bcm_rx_quick, 0, "do not m_devget in bcm driver");
#endif

MODULE_DEPEND(bcm, pci, 1, 1, 1);
MODULE_DEPEND(bcm, ether, 1, 1, 1);
MODULE_DEPEND(bcm, miibus, 1, 1, 1);
DRIVER_MODULE(bcm, pci, bcm_driver, bcm_devclass, 0, 0);
DRIVER_MODULE(miibus, bcm, miibus_driver, miibus_devclass, 0, 0);

#if __FreeBSD_version < 500000
#define ETHER_ALIGN	2
#define BPF_MTAP(_ifp,_m) do {                                  \
                        if ((_ifp)->if_bpf)                     \
                                bpf_mtap((_ifp), (_m));         \
                        } while (0)
#endif

/*
 *
 * PCI bus functions
 *
 */

static int
bcm_probe (device_t device)
{
	struct bcm_type		*t = bcm_devs;

	while(t->bcm_name != NULL) {
		if ((pci_get_vendor(device) == t->bcm_vid) &&
		    (pci_get_device(device) == t->bcm_did)) {
			device_set_desc(device, t->bcm_name);
			return(0);
		}
		t++;
	}

	return(ENXIO);
}

static int
bcm_attach(device_t device)
{
	int			unit	= device_get_unit(device);
	int			error	= 0;
	int			i;
	u_int32_t		epromdw[32];
	u_int8_t		*eprom	= (u_int8_t *)epromdw;
	struct bcm_softc 	*scp	= device_get_softc(device);
	struct ifnet 		*ifp;

#if __FreeBSD_version > 500000
	mtx_init(&scp->mtx, device_get_nameunit(device), MTX_NETWORK_LOCK,
	    MTX_DEF | MTX_RECURSE);
#endif
	scp->device = device;
	scp->unit = unit;

	/*
	 * Handle power management nonsense.
	 */
	if (pci_get_powerstate(device) != PCI_POWERSTATE_D0) {
		u_int32_t		membase, irq;

		/* Save important PCI config data. */
		membase = pci_read_config(device, BCM_PCI_MEMLO, 4);
		irq = pci_read_config(device, BCM_PCI_INTLINE, 4);

		/* Reset the power state. */
		printf("bcm%d: chip is is in D%d power mode "
		    "-- setting to D0\n", scp->unit,
		    pci_get_powerstate(device));

		pci_set_powerstate(device, PCI_POWERSTATE_D0);

		/* Restore PCI config data. */
		pci_write_config(device, BCM_PCI_MEMLO, membase, 4);
		pci_write_config(device, BCM_PCI_INTLINE, irq, 4);
	}

	/*
	 * Map control/status registers and IRQ.
	 */
	pci_enable_busmaster(device);

	scp->rid_memory = BCM_PCI_MEMLO;
	scp->res_memory = bus_alloc_resource(device, SYS_RES_MEMORY,
	    &scp->rid_memory, 0L, ~0L, 1, RF_ACTIVE);
	if (scp->res_memory == NULL) {
		printf ("bcm%d: could not map memory\n", scp->unit);
		error = ENXIO;
		goto fail_attach;
	}

	scp->bt = rman_get_bustag(scp->res_memory);
	scp->bh = rman_get_bushandle(scp->res_memory);
	scp->vh = (vm_offset_t)rman_get_virtual(scp->res_memory);

	scp->rid_irq = 0;
	scp->res_irq = bus_alloc_resource(device, SYS_RES_IRQ,
	    &scp->rid_irq, 0L, ~0L, 1, RF_SHAREABLE | RF_ACTIVE);
	if (scp->res_irq == NULL) {
		printf("bcm%d: could not map interrupt\n", scp->unit);
		error = ENXIO;
		goto fail_attach;
	}

	/*
	 * Get contents of EPROM
	 */
	for (i = 0; i < 32; i++)
	    epromdw[i] = REG_RD_OFFSET(scp, BCM_MEM_EPROM + 4*i);

	/* Permanent ether address */
	scp->arpcom.ac_enaddr[0] = eprom[79];
	scp->arpcom.ac_enaddr[1] = eprom[78];
	scp->arpcom.ac_enaddr[2] = eprom[81];
	scp->arpcom.ac_enaddr[3] = eprom[80];
	scp->arpcom.ac_enaddr[4] = eprom[83];
	scp->arpcom.ac_enaddr[5] = eprom[82];

	scp->miphy = eprom[90] & 0x1f;
	scp->miport = (eprom[90] >> 14) & 0x1;
	scp->phy_reset = 0;

	/*
	 * A Broadcom chip was detected. Inform the world.
	 */
	printf("bcm%d: Ethernet address: %6D\n", scp->unit,
	    scp->arpcom.ac_enaddr, ":");

	/*
	 * Allocate the parent bus DMA tag appropriate for PCI.
	 */
	error = bus_dma_tag_create(NULL,	/* parent */
			1, 0,			/* alignment, boundary */
			BUS_SPACE_MAXADDR_32BIT,/* lowaddr */
			BUS_SPACE_MAXADDR,	/* highaddr */
			NULL, NULL,		/* filter, filterarg */
			MAXBSIZE,		/* maxsize */
			BUS_SPACE_UNRESTRICTED,	/* nsegments */
			BUS_SPACE_MAXSIZE_32BIT,/* maxsegsize */
			BUS_DMA_ALLOCNOW,	/* flags */
#if __FreeBSD_version > 500000
			busdma_lock_mutex, &Giant,
#endif
			&scp->tag_parent);
	if (error)
		goto fail_attach;

	/*
	 * Allocate a DMA tag for DMAing mbufs with.
	 */
	error = bus_dma_tag_create(scp->tag_parent,	/* parent */
			1, 0,			/* alignment, boundary */
			BUS_SPACE_MAXADDR,	/* lowaddr */
			BUS_SPACE_MAXADDR,	/* highaddr */
			NULL, NULL,		/* filter, filterarg */
			MCLBYTES, 1,		/* maxsize,nsegments */
			BUS_SPACE_MAXSIZE_32BIT,/* maxsegsize */
			0,			/* flags */
#if __FreeBSD_version > 500000
			busdma_lock_mutex, &Giant,
#endif
			&scp->tag_mbuf);
	if (error)
		goto fail_attach;

	/*
	 * Allocate tags and memory for the TX DMA descriptor lists.
	 */
	error = bus_dma_tag_create(scp->tag_parent,	/* parent */
			DMARINGALIGN,		/* alignment */
			DMARINGALIGN,		/* boundary */
			BUS_SPACE_MAXADDR,	/* lowaddr */
			BUS_SPACE_MAXADDR,	/* highaddr */
			NULL, NULL,		/* filter, filterarg */
			DMAMAXRINGSZ, 1,	/* maxsize,nsegments */
			BUS_SPACE_MAXSIZE_32BIT,/* maxsegsize */
			0,			/* flags */
#if __FreeBSD_version > 500000
			busdma_lock_mutex, &Giant,
#endif
			&scp->tag_tx);
	if (error)
		goto fail_attach;

	error = bus_dmamem_alloc(scp->tag_tx, (void *)&scp->desc_tx,
	    BUS_DMA_NOWAIT, &scp->dmamap_tx);
	if (error) {
		printf("bcm%d: no memory for TX DMA descriptors\n", scp->unit);
		bus_dma_tag_destroy(scp->tag_tx);
		scp->tag_tx = NULL;
		goto fail_attach;
	}

	error = bus_dmamap_load(scp->tag_tx, scp->dmamap_tx, scp->desc_tx,
	    sizeof(struct bcm_dma_desc), bcm_ring_map, &scp->paddr_tx, 0);
	if (error) {
		printf("bcm%d: cannot get address of the TX ring\n", scp->unit);
		bus_dmamem_free(scp->tag_tx, scp->desc_tx, scp->dmamap_tx);
		bus_dma_tag_destroy(scp->tag_tx);
		scp->tag_tx = NULL;
		goto fail_attach;
	}
	bzero(scp->desc_tx, DMAMAXRINGSZ);
	bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx, BUS_DMASYNC_PREREAD);

	/*
	 * Allocate tags and memory for the RX DMA descriptor lists.
	 */
	error = bus_dma_tag_create(scp->tag_parent,	/* parent */
			DMARINGALIGN,		/* alignment */
			DMARINGALIGN,		/* boundary */
			BUS_SPACE_MAXADDR,	/* lowaddr */
			BUS_SPACE_MAXADDR,	/* highaddr */
			NULL, NULL,		/* filter, filterarg */
			DMAMAXRINGSZ, 1,	/* maxsize,nsegments */
			BUS_SPACE_MAXSIZE_32BIT,/* maxsegsize */
			0,			/* flags */
#if __FreeBSD_version > 500000
			busdma_lock_mutex, &Giant,
#endif
			&scp->tag_rx);
	if (error)
		goto fail_attach;

	error = bus_dmamem_alloc(scp->tag_rx, (void *)&scp->desc_rx,
	    BUS_DMA_NOWAIT, &scp->dmamap_rx);
	if (error) {
		printf("bcm%d: no memory for RX DMA descriptors\n", scp->unit);
		bus_dma_tag_destroy(scp->tag_rx);
		scp->tag_rx = NULL;
		goto fail_attach;
	}

	error = bus_dmamap_load(scp->tag_rx, scp->dmamap_rx, scp->desc_rx,
	    sizeof(struct bcm_dma_desc), bcm_ring_map, &scp->paddr_rx, 0);
	if (error) {
		printf("bcm%d: cannot get address of the RX ring\n", scp->unit);
		bus_dmamem_free(scp->tag_rx, scp->desc_rx, scp->dmamap_rx);
		bus_dma_tag_destroy(scp->tag_rx);
		scp->tag_rx = NULL;
		goto fail_attach;
	}
	bzero(scp->desc_rx, DMAMAXRINGSZ);
	bus_dmamap_sync(scp->tag_rx, scp->dmamap_rx, BUS_DMASYNC_PREREAD);
    	scp->rxoffset = sizeof(struct bcm_rxh);
	if (MCLBYTES < scp->rxoffset + BCM_BUFFER_SIZE) {
	    	printf("bcm%d: MCLBYTES too small to hold RX header\n",
		    scp->unit);
		error = ENXIO;
		goto fail_attach;
	}

	/*
	 * Reset the chip and PHY. We have to do this here
	 * so that the PHY is turned on. But we do not enable
	 * them - that's done in bcm_init once we got interrupt
	 * handlers and buffer rings set up.
	 */
	bcm_chip_reset(scp);

	/* Do MII setup */
	if (mii_phy_probe(device, &scp->miibus,
	    bcm_ifmedia_upd, bcm_ifmedia_sts)) {
		printf("bcm%d: MII without any phy!\n", scp->unit);
		error = ENXIO;
		goto fail_attach;
	}

	/*
	 * Ethernet interface setup
	 */
	ifp = &scp->arpcom.ac_if;
	ifp->if_softc = scp;
	ifp->if_unit = unit;
	ifp->if_name = "bcm";
	ifp->if_mtu = ETHERMTU;
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
	ifp->if_ioctl = bcm_ioctl;
	ifp->if_output = ether_output;
	ifp->if_start = bcm_start;
	ifp->if_watchdog = bcm_watchdog;
	ifp->if_init = bcm_init;
	ifp->if_baudrate = 10000000;
	ifp->if_snd.ifq_maxlen = IFQ_MAXLEN;

	callout_handle_init(&scp->stat_ch);
#if __FreeBSD_version > 500000
	ether_ifattach(ifp, scp->arpcom.ac_enaddr);
#else
	ether_ifattach(ifp, ETHER_BPF_SUPPORTED);
#endif

	/*
	 * Hook interrupt last to avoid having to lock softc
	 */
	error = bus_setup_intr(device, scp->res_irq, INTR_TYPE_NET,
	    bcm_intr, scp, &scp->intrhand);
	if (error) {
		printf("bcm%d: could not set up irq\n", scp->unit);
		goto fail_attach;
	}

	return 0;

fail_attach:
	bcm_detach(device);
	return (error);
}

static int
bcm_detach(device_t device)
{
	struct bcm_softc 	*scp	= device_get_softc(device);
	struct ifnet 		*ifp;

#if __FreeBSD_version > 500000
BCM_XXX_PRINTF("bcm_detach:\n");
	KASSERT(mtx_initialized(&scp->mtx), ("bcm mutex not initialized"));
#endif
	BCM_LOCK(scp);

	ifp = &scp->arpcom.ac_if;

#if __FreeBSD_version > 500000
	/* These should only be active if attach succeeded */
	if (device_is_attached(device)) {
		bcm_stop(scp);
		ether_ifdetach(ifp);
	}
	if (scp->miibus)
		device_delete_child(device, scp->miibus);
#else
	bcm_stop(scp);
	ether_ifdetach(ifp, ETHER_BPF_SUPPORTED);
#endif

	bus_generic_detach(device);

	if (scp->intrhand)
		bus_teardown_intr(device, scp->res_irq, scp->intrhand);
	if (scp->res_irq != 0) {
		bus_deactivate_resource(device, SYS_RES_IRQ,
			scp->rid_irq, scp->res_irq);
		bus_release_resource(device, SYS_RES_IRQ,
			scp->rid_irq, scp->res_irq);
		scp->res_irq = 0;
	}
	if (scp->res_memory) {
		bus_deactivate_resource(device, SYS_RES_MEMORY,
			scp->rid_memory, scp->res_memory);
		bus_release_resource(device, SYS_RES_MEMORY,
			scp->rid_memory, scp->res_memory);
		scp->res_memory = 0;
	}
	if (scp->tag_rx) {
		bus_dmamap_unload(scp->tag_rx, scp->dmamap_rx);
		bus_dmamem_free(scp->tag_rx, scp->desc_rx, scp->dmamap_rx);
		bus_dma_tag_destroy(scp->tag_rx);
		scp->tag_rx = NULL;
	}
	if (scp->tag_tx) {
		bus_dmamap_unload(scp->tag_tx, scp->dmamap_tx);
		bus_dmamem_free(scp->tag_tx, scp->desc_tx, scp->dmamap_tx);
		bus_dma_tag_destroy(scp->tag_tx);
		scp->tag_tx = NULL;
	}
	if (scp->tag_mbuf) {
		bus_dma_tag_destroy(scp->tag_mbuf);
		scp->tag_mbuf = NULL;
	}
	if (scp->tag_parent) {
		bus_dma_tag_destroy(scp->tag_parent);
		scp->tag_parent = NULL;
	}

	scp->phy_reset = 0;

	BCM_UNLOCK(scp);
#if __FreeBSD_version > 500000
	mtx_destroy(&scp->mtx);
#endif

	return (0);
}

/*
 * Stop all chip I/O so that the kernel's probe routines don't
 * get confused by errant DMAs when rebooting.
 */
static void
bcm_shutdown(device_t device)
{
	struct bcm_softc 	*scp	= device_get_softc(device);

	BCM_LOCK(scp);

BCM_XXX_PRINTF("bcm_shutdown:\n");
	bcm_stop(scp);

	BCM_UNLOCK(scp);
    	return;
}

/*
 *
 * ifnet functions
 *
 */

/*
 * Set media options.
 */
static int
bcm_ifmedia_upd(struct ifnet *ifp)
{
	struct bcm_softc	*scp = ifp->if_softc;
	struct mii_data		*mii;

#if __FreeBSD_version > 500000
BCM_XXX_PRINTF("bcm_if_media_upd: starting\n");
#endif
	BCM_LOCK(scp);

	mii = device_get_softc(scp->miibus);
	mii_mediachg(mii);

	BCM_UNLOCK(scp);

	return(0);
}

/*
 * Media status
 */
static void
bcm_ifmedia_sts(struct ifnet *ifp, struct ifmediareq *ifmr)
{
	struct bcm_softc	*scp = ifp->if_softc;
	struct mii_data		*mii;

#if __FreeBSD_version > 500000
BCM_XXX_PRINTF("bcm_if_media_sts: starting\n");
#endif
	BCM_LOCK(scp);

	mii = device_get_softc(scp->miibus);

	mii_pollstat(mii);
	ifmr->ifm_active = mii->mii_media_active;
	ifmr->ifm_status = mii->mii_media_status;

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_init(void *arg)
{
	struct bcm_softc	*scp = (struct bcm_softc *)arg;
	struct ifnet		*ifp;

	BCM_LOCK(scp);
	ifp = &scp->arpcom.ac_if;

BCM_XXX_PRINTF("bcm_init: if_flags 0x%x\n", ifp->if_flags);
	if (ifp->if_flags & IFF_RUNNING) {
BCM_XXX_PRINTF("bcm_init: already running\n");
		BCM_UNLOCK(scp);
		return;
	}

	bcm_stop(scp);
	bcm_chip_reset(scp);

	/*
	 * Init descriptors
	 */
	bcm_ring_tx_init(scp);
	if (bcm_ring_rx_init(scp) == ENOBUFS) {
		printf("bcm%d: initialization failed: no "
			"memory for rx buffers\n", scp->unit);
		bcm_stop(scp);
		BCM_UNLOCK(scp);
		return;
	}

	/*
	 * Sort out starting the chip up
	 */
	bcm_chip_rxfilter(scp);
	bcm_chip_enable(scp);
	bcm_ifmedia_upd(ifp);

	ifp->if_flags |= IFF_RUNNING;
	ifp->if_flags &= ~IFF_OACTIVE;

	scp->stat_ch = timeout(bcm_tick, scp, hz);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Interrupt routine
 */
static void
bcm_intr(void *arg)
{
	struct bcm_softc	*scp = (struct bcm_softc *)arg;
	struct ifnet		*ifp;
	u_int32_t		intstatus, intmask, inttaken;
	
	BCM_LOCK(scp);

	ifp = &scp->arpcom.ac_if;

	while (1) {
	    	inttaken = 0;
		intstatus = REG_RD(scp, intstatus);
		intmask = REG_RD(scp, intmask);

		/* defer unsolicited interrupts */
		intstatus &= intmask;
BCM_XXX_PRINTF("bcm_intr: intstatus 0x%x\n", intstatus);

		/* if not for us */
		if (intstatus == 0)
			break;

		if (intstatus & I_RI) {
			bcm_ring_rx_eof(scp);
			inttaken |= I_RI;
		}

		if (intstatus & I_XI) {
			bcm_ring_tx_eof(scp);
			inttaken |= I_XI;
		}

		if (intstatus & I_ERRORS) {
printf("bcm%d: ", scp->unit);
if (intstatus & I_PC) {
	ifp->if_oerrors++;
	ifp->if_ierrors++;
	printf("descriptor error.\n");
	inttaken |= I_PC;
}
if (intstatus & I_PD) {
	ifp->if_oerrors++;
	ifp->if_ierrors++;
	printf("data error.\n");
	inttaken |= I_PD;
}
if (intstatus & I_DE) {
	ifp->if_oerrors++;
	ifp->if_ierrors++;
	printf("descriptor protocol error.\n");
	inttaken |= I_DE;
}
if (intstatus & I_RU) {
    	u_int32_t	status;
	ifp->if_ierrors++;
	printf("receive descriptor underflow.");
	status = REG_RD(scp, dmaregs.rcvstatus);
	printf(" con_rx = %d", scp->con_rx);
	printf(", macCurrent %d, status 0x%x, error 0x%x\n",
	    (status & RS_CD_MASK)/sizeof(struct bcm_dma_desc),
	    status & RS_RS_MASK,
	    status & RS_RE_MASK);
	inttaken |= I_RU;
}
if (intstatus & I_RO) {
	ifp->if_ierrors++;
	printf("\treceive fifo underflow\n");
	inttaken |= I_RO;
}
if (intstatus & I_XU) {
	ifp->if_oerrors++;
	printf("\ttransmit fifo underflow\n");
	inttaken |= I_XU;
}

			ifp->if_flags &= ~IFF_RUNNING;
			bcm_init(scp);
		}
#if BCM_XXX_INTRCHECK
if (inttaken != intstatus)
	printf("bcm_intr: should do another\n");
#endif /* BCM_XXX_INTRCHECK */
		REG_WR(scp, intstatus, inttaken);
	}

#if XXX
	/* Check RX+TX ring producer/consumer */
	if (ifp->if_flags & IFF_RUNNING) {
		bcm_ring_rx_eof(scp);
		bcm_ring_tx_eof(scp);
		if (ifp->if_snd.ifq_head != NULL)
			bcm_start(ifp);
	}
#endif

	BCM_UNLOCK(scp);
	return;
}

/*
 * IOCTL handler
 */
static int
bcm_ioctl(struct ifnet *ifp, u_long command, caddr_t data)
{
	struct bcm_softc	*scp = ifp->if_softc;
	struct ifreq		*ifr = (struct ifreq *)data;
	struct mii_data		*mii;
	struct bcm_mib		mib;
	int			error = 0;

	BCM_LOCK(scp);

	switch(command) {
	case SIOCSIFFLAGS:
		if (ifp->if_flags & IFF_UP) {
			if (ifp->if_flags & IFF_RUNNING)
				bcm_chip_rxfilter(scp);
			else
				bcm_init(scp);
		} else {
			if (ifp->if_flags & IFF_RUNNING)
				bcm_stop(scp);
		}
		error = 0;
		break;
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		if (ifp->if_flags & IFF_RUNNING) {
			bcm_chip_rxfilter(scp);
		}
		error = 0;
		break;
	case SIOCGIFMEDIA:
	case SIOCSIFMEDIA:
		mii = device_get_softc(scp->miibus);
		error = ifmedia_ioctl(ifp, ifr, &mii->mii_media, command);
		break;
	case SIOCGBCMSTATS:
		bcm_chip_get_stats(scp, &mib);
	    	error = copyout(&mib, ifr->ifr_data, sizeof(mib));
		break;
#if __FreeBSD_version > 500000
	default:
		error = ether_ioctl(ifp, command, data);
		break;
#else
	case SIOCSIFADDR:
	case SIOCGIFADDR:
	case SIOCSIFMTU:
		error = ether_ioctl(ifp, command, data);
		break;

	default:
		error = EINVAL;
		break;
#endif
	}

	BCM_UNLOCK(scp);

	return(error);
}

/*
 * Set up to transmit a packet
 */
static void
bcm_start(struct ifnet *ifp)
{
	struct bcm_softc	*scp = ifp->if_softc;
	struct mbuf		*m_head = NULL;
	int			idx;

	BCM_LOCK(scp);

	ifp = &scp->arpcom.ac_if;

	if (ifp->if_flags & IFF_OACTIVE) {
		BCM_UNLOCK(scp);
		return;
	}

	idx = scp->idx_tx;
	while (scp->desc_tx_bis[idx].mbuf == NULL) {
		IF_DEQUEUE(&ifp->if_snd, m_head);
		if (m_head == NULL)
			break;

		/*
		 * Pack the data into the transmit ring. If we
		 * don't have room, set the OACTIVE flag and wait
		 * for the NIC to drain the ring.
		 */
		if (bcm_ring_tx_encap(scp, m_head, &idx)) {
			IF_PREPEND(&ifp->if_snd, m_head);
			ifp->if_flags |= IFF_OACTIVE;
			break;
		}

		/*
		 * If there's a BPF listener, bounce a copy of this frame
		 * to him.
		 */
		BPF_MTAP(ifp, m_head);
	}
	/* Transmit */
	scp->idx_tx = idx;
	REG_WR(scp, dmaregs.xmtptr, (idx * sizeof(struct bcm_dma_desc)));
	REG_WR(scp, dmaregs.xmtptr, (idx * sizeof(struct bcm_dma_desc)));

	/*
	 * Set a timeout in case the chip goes out to lunch.
	 */
	ifp->if_timer = 5;
		
	BCM_UNLOCK(scp);
    	return;
}

/*
 * Stop activity on the interface
 */
static void
bcm_stop(struct bcm_softc *scp)
{
	struct ifnet		*ifp;

#if __FreeBSD_version > 500000
BCM_XXX_PRINTF("bcm_stop:\n");
#endif
	BCM_LOCK(scp);

	ifp = &scp->arpcom.ac_if;

	bcm_chip_halt(scp);

	bcm_ring_tx_free(scp);
	bcm_ring_rx_free(scp);

	untimeout(bcm_tick, scp, scp->stat_ch);

	ifp->if_flags &= ~(IFF_RUNNING | IFF_OACTIVE);

	BCM_UNLOCK(scp);
    	return;
}

static void
bcm_tick(void *arg)
{
	struct bcm_softc	*scp = (struct bcm_softc *)arg;
	struct mii_data		*mii;

	BCM_LOCK(scp);

	mii = device_get_softc(scp->miibus);
	mii_tick(mii);
	
#if BCM_XXX_EMAC_FLOWCONTROL
	bcm_chip_flow(scp, bcm_mi_readphy(scp, PHY_AN_AD_REG),
	    bcm_mi_readphy(scp, PHY_LINK_PARTNER_ABILITY_REG));
#endif /* BCM_XXX_EMAC_FLOWCONTROL */

	scp->stat_ch = timeout(bcm_tick, scp, hz);

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_watchdog(struct ifnet *ifp)
{
	struct bcm_softc	*scp = ifp->if_softc;

	BCM_LOCK(scp);

	printf("bcm%d: watchdog timeout\n", scp->unit);
#if BCM_XXX_INTRCHECK
	{
		u_int32_t intstatus;
		intstatus = REG_RD(scp, intstatus);
		printf("bcm_watchdog: intstatus = 0x%x\n", intstatus);
		if (intstatus) {
			bcm_intr(scp);
			BCM_UNLOCK(scp);
			return;
		}
	}
#endif /* BCM_XXX_INTRCHECK */

	ifp->if_oerrors++;
	ifp->if_flags &= ~IFF_RUNNING;
	bcm_init(scp);

	BCM_UNLOCK(scp);
	return;
}

/*
 *
 * Buffer ring handling
 *
 * MUST BE LOCKED
 *
 */

/*
 * XXX when writing rx handler, if BCM_XXX_LAZYRXFC > 1 have so
 * XXX code to count number of packets rx'd at once and
 * XXX flood ping the machine!
 */

/*
 * Pass a received mbuf up the stack
 */
static void
bcm_ring_rx_eof(struct bcm_softc *scp)
{
	struct ifnet 		*ifp;
	u_int32_t		status;
	int			con, macCurrent;

	ifp = &scp->arpcom.ac_if;

	status = REG_RD(scp, dmaregs.rcvstatus);
	macCurrent = (status & RS_CD_MASK)/sizeof(struct bcm_dma_desc);

/* XXX error check etc. */

	/*
	 * Walk the ring of buffers
	 */
	con = scp->con_rx;
	while (con != macCurrent) {
		struct mbuf	*m;
		struct bcm_rxh	*rxh;
		int		j = 0, len, flags;

		m = scp->desc_rx_bis[con].mbuf;
		rxh = mtod(m, struct bcm_rxh *);
		do {
	/* XXX		DELAY(1); */
			bus_dmamap_sync(scp->tag_mbuf,
			    scp->desc_rx_bis[con].dmamap,
			    BUS_DMASYNC_POSTWRITE);
			len = rxh->len;
		} while (len == 0 && j++ < 5);
if (len == 0) printf("eek j=%d, macCurrent %d, con %d\n", j, macCurrent, con);
		scp->desc_rx_bis[con].mbuf = NULL;
		bus_dmamap_unload(scp->tag_mbuf, scp->desc_rx_bis[con].dmamap);
		bus_dmamap_destroy(scp->tag_mbuf, scp->desc_rx_bis[con].dmamap);
		flags = rxh->flags;
		len -= ETHER_CRC_LEN;

		if ((len > BCM_BUFFER_SIZE) ||
		    (flags & RXF_ERRORS)) {
			ifp->if_ierrors++;
			if (flags & RXF_RXER)
				ifp->if_collisions++;
			bcm_ring_rx_newbuf(scp, con, m);
			continue;
		}

		/*
		 * No errors; remove the MAC header and
		 * receive the packet.
		 */
#ifdef __i386__
		/*
		 * On the x86 we do not have alignment problems, so try to
		 * allocate a new buffer for the receive ring, and pass up
		 * the one where the packet is already, saving the expensive
		 * copy done in m_devget().
		 * If we are on an architecture with alignment problems, or
		 * if the allocation fails, then use m_devget and leave the
		 * existing buffer in the receive ring.
		 */
		if (bcm_rx_quick && bcm_ring_rx_newbuf(scp, con, NULL) == 0) {
			struct mbuf		*m0;

/* printf("pullup %d, %d\n", scp->rxoffset+ETHER_HDR_LEN, MHLEN); */
			m->m_pkthdr.len = m->m_len = len + scp->rxoffset;
#if __FreeBSD_version > 500000
			m0 = m_pullup(m, scp->rxoffset + ETHER_HDR_LEN);
			if (m0 == NULL) {
				ifp->if_ierrors++;
				continue;
			}
			m = m0;
#endif
/* printf("pulled okay - length = %d\n", m->m_len); */
			m_adj(m, scp->rxoffset);
/* printf("m_adj - length = %d\n", m->m_len); */
		} else
#endif
		{
			struct mbuf		*m0;
			m0 = m_devget(mtod(m, char *)+scp->rxoffset, len,
				ETHER_ALIGN, ifp, NULL);
			bcm_ring_rx_newbuf(scp, con, m);
			if (m0 == NULL) {
				ifp->if_ierrors++;
				continue;
			}
			m = m0;
		}
{
#if 0
    u_int8_t *mm;

    mm = mtod(m, u_int8_t *);
    printf("rx packet after, len %d, DST:%6D, SRC:%6D, TYPE:%4x\n",
    	len, mm, ":", mm+6, ":", *(mm+12));
    if (bcmp(mm, scp->arpcom.ac_enaddr, 6)) {
	printf("runt before if_input? len %d\n", len);
	BCM_XXX_RXDUMP(mtod(m, u_int8_t *), len);
	printf("\n");
    }
#endif
}

		ifp->if_ipackets++;
		m->m_pkthdr.rcvif = ifp;
#if __FreeBSD_version > 500000
		(*ifp->if_input)(ifp, m);
#else
		ether_input(ifp, NULL, m);
#endif

		BCM_INC(con, BCM_RX_LIST_CNT);
#if XXX
		status = REG_RD(scp, dmaregs.rcvstatus);
		macCurrent = (status & RS_CD_MASK)/sizeof(struct bcm_dma_desc);
	    	if (con == macCurrent)
			break;
#endif
	}

	scp->con_rx = con;

	return;
}

/*
 * Allocate a ring of mbufs for RX use
 */
static int
bcm_ring_rx_init(struct bcm_softc *scp)
{
	int			i;

	bzero(scp->desc_rx, DMAMAXRINGSZ);
	for (i = 0; i < BCM_RX_LIST_CNT; i++) {
		if (bcm_ring_rx_newbuf(scp, i, NULL) == ENOBUFS)
			return(ENOBUFS);
	}
	bus_dmamap_sync(scp->tag_rx, scp->dmamap_rx, BUS_DMASYNC_PREREAD);

	REG_WR(scp, dmaregs.rcvptr, (i * sizeof(struct bcm_dma_desc)));

	scp->con_rx = 0;

	return (0);
}

static void
bcm_ring_rx_free(struct bcm_softc *scp)
{
	int			i;

	for (i = 0; i < BCM_RX_LIST_CNT; i++) {
		if (scp->desc_rx_bis[i].mbuf != NULL) {
			m_freem(scp->desc_rx_bis[i].mbuf);
			scp->desc_rx_bis[i].mbuf = NULL;
			bus_dmamap_unload(scp->tag_mbuf,
			    scp->desc_rx_bis[i].dmamap);
			bus_dmamap_destroy(scp->tag_mbuf,
				scp->desc_rx_bis[i].dmamap);
		}
	}
	bzero(scp->desc_rx, DMAMAXRINGSZ);
	bus_dmamap_sync(scp->tag_rx, scp->dmamap_rx, BUS_DMASYNC_PREREAD);

	return;
}

/*
 * Initialize an RX descriptor and attach an MBUF cluster. Allocate
 * a new cluster if m is NULL, or re-use an existing one.
 */
static int
bcm_ring_rx_newbuf(struct bcm_softc *scp, int c, struct mbuf *m)
{
	volatile struct bcm_rxh	*rxh;
	u_int32_t		ctrl;

	if ((c < 0) || (c >= BCM_RX_LIST_CNT))
		return (EINVAL);

	if (m == NULL) {
		m = m_getcl(M_DONTWAIT, MT_DATA, M_PKTHDR);
		if (m == NULL)
			return (ENOBUFS);
	} else
		m->m_data = m->m_ext.ext_buf;

	/*
	 * Clear the mbuf and set it up for DMAing into
	 */
	scp->desc_rx_bis[c].mbuf = m;
	rxh = mtod(m, struct bcm_rxh *);
	rxh->flags = 0;
	rxh->len = 0;
	scp->dma_rx = c;

	/*
	 * Attach the mbuf into the DMA descriptor ring
	 */
	ctrl = BCM_BUFFER_SIZE;
	if (c == BCM_RX_LIST_CNT-1)
		ctrl |= CTRL_EOT;
	scp->desc_rx[c].ctrl = ctrl;
	bus_dmamap_create(scp->tag_mbuf, 0, &scp->desc_rx_bis[c].dmamap);
	bus_dmamap_sync(scp->tag_mbuf, scp->desc_rx_bis[c].dmamap,
	    BUS_DMASYNC_PREWRITE);
	bus_dmamap_load(scp->tag_mbuf, scp->desc_rx_bis[c].dmamap,
	    mtod(m, void *), MCLBYTES, bcm_ring_rx_map, scp, 0);
	bus_dmamap_sync(scp->tag_rx, scp->dmamap_rx, BUS_DMASYNC_PREREAD);

	return(0);
}

static void
bcm_ring_rx_map(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
	struct bcm_softc	*scp = arg;

    	scp->desc_rx[scp->dma_rx].addr = segs->ds_addr + BCM_SBBASE_PCIDMA;

	return;
}

/*
 * Encapsulate an mbuf chain in a descriptor by coupling the mbuf data
 * pointers to the fragment pointers.
 */
static int
bcm_ring_tx_encap(struct bcm_softc *scp, struct mbuf *m_head, int *idx)
{
	struct mbuf		*m;
	u_int32_t		ctrl;
	int			error = 0;
	int			curFrag, lastFrag, cnt, chainlen;

	BCM_LOCK(scp);

	/*
	 * If there's no way we can send any packets, return now.
	 */
	if (BCM_TX_LIST_CNT - scp->cnt_tx < 2) {
		error = ENOBUFS;
		goto tx_encap_fail;
	}

#if __FreeBSD_version > 500000
	/*
	 * Count the number of frags in this chain to see if
	 * we need to m_defrag.  Since the descriptor list is shared
	 * by all packets, we'll m_defrag long chains so that they
	 * do not use up the entire list, even if they would fit.
	 */
	chainlen = 0;
	for (m = m_head; m != NULL; m = m->m_next)
		chainlen++;

	if ((chainlen > BCM_TX_LIST_CNT / 4) ||
	    ((BCM_TX_LIST_CNT - (chainlen + scp->cnt_tx)) < 2)) {
		m = m_defrag(m_head, M_DONTWAIT);
		if (m == NULL) {
			error = ENOBUFS;
			goto tx_encap_fail;
		}
		m_head = m;
	}
#endif

	/*
 	 * Start packing the mbufs in this chain into
	 * the fragment pointers. Stop when we run out
 	 * of fragments or hit the end of the mbuf chain.
	 */
	m = m_head;
	curFrag = lastFrag = *idx;
	cnt = 0;
BCM_XXX_PRINTF("bcm_ring_tx_encap: curFrag %d, lastFrag %d\n", curFrag, lastFrag);

	for (m = m_head; m != NULL; m = m->m_next) {
		if (m->m_len != 0) {
		    	scp->dma_tx = curFrag;
			if ((BCM_TX_LIST_CNT -
			    (scp->cnt_tx + cnt)) < 2) {
				error = ENOBUFS;
				goto tx_encap_fail;
			}
			ctrl = m->m_len & CTRL_BC_MASK;
			ctrl |= CTRL_IOC;
			if (cnt == 0)
				ctrl |= CTRL_SOF;
			if (curFrag == BCM_TX_LIST_CNT-1)
				ctrl |= CTRL_EOT;
			scp->desc_tx[curFrag].ctrl = ctrl;

			bus_dmamap_create(scp->tag_mbuf, 0,
			    &scp->desc_tx_bis[curFrag].dmamap);
			bus_dmamap_load(scp->tag_mbuf,
			    scp->desc_tx_bis[curFrag].dmamap,
			    mtod(m, void *), m->m_len, bcm_ring_tx_map,
			    scp, 0);
			bus_dmamap_sync(scp->tag_mbuf,
			    scp->desc_tx_bis[curFrag].dmamap,
			    BUS_DMASYNC_PREREAD);
			bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx,
			    BUS_DMASYNC_PREREAD);

			lastFrag = curFrag;
			BCM_INC(curFrag, BCM_TX_LIST_CNT);
			cnt++;
BCM_XXX_PRINTF("bcm_ring_tx_encap: curFrag %d, lastFrag %d, cnt %d\n", curFrag, lastFrag, cnt);
BCM_XXX_TXDUMP(mtod(m, u_int8_t *), m->m_len);
		}
	}

	if (m != NULL) {
		error = ENOBUFS;
		goto tx_encap_fail;
	}

	/*
	 * Mark as last fragment and ensure that the memory is
	 * coherant
	 */
	ctrl = scp->desc_tx[lastFrag].ctrl;
	ctrl |= CTRL_EOF;
	scp->desc_tx[lastFrag].ctrl = ctrl;
	bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx, BUS_DMASYNC_PREREAD);

	scp->desc_tx_bis[lastFrag].mbuf = m_head;

	scp->cnt_tx += cnt;
	*idx = curFrag;

tx_encap_fail:
	BCM_UNLOCK(scp);
	return (error);
}

static void
bcm_ring_tx_eof(struct bcm_softc *scp)
{
	struct ifnet		*ifp;
	int			idx;
	u_int32_t		chipIdx, status;

	BCM_LOCK(scp);

	ifp = &scp->arpcom.ac_if;

	status = REG_RD(scp, dmaregs.xmtstatus);
	chipIdx = status & XS_CD_MASK;
	status = status & XS_XE_MASK;
	chipIdx = chipIdx / sizeof(struct bcm_dma_desc);

	/*
	 * Examine status for errors
	 */
	if (status)
		ifp->if_oerrors++;
if (status)
	printf("had tx error: 0x%x\n", status);

	/*
	 * Go through our tx list and free mbufs for those
	 * frames that have been transmitted.
	 */
	for (idx = scp->con_tx; scp->cnt_tx > 0;
	    scp->cnt_tx--, BCM_INC(idx, BCM_TX_LIST_CNT)) {

		if (idx == chipIdx)
			break;

		if (scp->desc_tx_bis[idx].mbuf != NULL) {
			ifp->if_opackets++;
			m_freem(scp->desc_tx_bis[idx].mbuf);
			scp->desc_tx_bis[idx].mbuf = NULL;
		}
		scp->desc_tx[idx].ctrl = 0;
		scp->desc_tx[idx].addr = 0;
		bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx,
		    BUS_DMASYNC_PREREAD);
		bus_dmamap_unload(scp->tag_mbuf, scp->desc_tx_bis[idx].dmamap);
		bus_dmamap_destroy(scp->tag_mbuf, scp->desc_tx_bis[idx].dmamap);
	}

	/*
	 * Did we free up some buffers?
	 */
	if (idx != scp->con_tx) {
		scp->con_tx = idx;
		ifp->if_flags &= ~IFF_OACTIVE;
	}

	ifp->if_timer = (scp->cnt_tx == 0) ? 0 : 5;

BCM_XXX_PRINTF("bcm_ring_tx_eof: idx %d, cnt %d con %d\n", scp->idx_tx, scp->cnt_tx, scp->con_tx);

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_ring_tx_init(struct bcm_softc *scp)
{
	int			i;

    	BCM_LOCK(scp);

	bzero(scp->desc_tx, DMAMAXRINGSZ);
	bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx, BUS_DMASYNC_PREREAD);

	for (i = 0; i < BCM_TX_LIST_CNT; i++) {
		scp->desc_tx_bis[i].mbuf = NULL;
		scp->desc_tx_bis[i].dmamap = NULL;
	}
	scp->idx_tx = scp->cnt_tx = scp->con_tx = 0;

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_ring_tx_map(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
	struct bcm_softc	*scp = arg;

    	scp->desc_tx[scp->dma_tx].addr = segs->ds_addr + BCM_SBBASE_PCIDMA;

	return;
}

/*
 * Free the TX list buffers.
 */
static void
bcm_ring_tx_free(struct bcm_softc *scp)
{
	int			i;

	BCM_LOCK(scp);


	for (i = 0; i < BCM_TX_LIST_CNT; i++) {
		if (scp->desc_tx_bis[i].mbuf != NULL) {
			m_freem(scp->desc_tx_bis[i].mbuf);
			scp->desc_tx_bis[i].mbuf = NULL;
			bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx,
			    BUS_DMASYNC_PREREAD);
			bus_dmamap_unload(scp->tag_mbuf,
			    scp->desc_tx_bis[i].dmamap);
			bus_dmamap_destroy(scp->tag_mbuf,
				scp->desc_tx_bis[i].dmamap);
		}
	}
	bzero(scp->desc_tx, DMAMAXRINGSZ);
	bus_dmamap_sync(scp->tag_tx, scp->dmamap_tx, BUS_DMASYNC_PREREAD);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Get physical address of ring
 */
static void
bcm_ring_map(void *arg, bus_dma_segment_t *segs, int nseg, int error)
{
	u_int32_t *p;

	p = arg;
	*p = segs->ds_addr;

	return;
}

/*
 *
 * MII functions
 *
 */

static int
bcm_mi_readphy(struct bcm_softc *scp, u_int32_t reg)
{
	int			data;

	BCM_LOCK(scp);

	/* clear mii_int */
	REG_WR(scp, emacintstatus, EI_MII);

	/* issue the read */
	REG_WR(scp, mdiodata, (MD_SB_START | MD_OP_READ |
		(scp->miphy << MD_PMD_SHIFT)
		| (reg << MD_RA_SHIFT) | MD_TA_VALID));

	/* wait for it to complete */
	SPINWAIT(((REG_RD(scp, emacintstatus) & EI_MII) == 0), 100);
	if ((REG_RD(scp, emacintstatus) & EI_MII) == 0) {
		printf("bcm%d: PHY read timed out\n", scp->unit);
	}

	data = REG_RD(scp, mdiodata) & MD_DATA_MASK;

	BCM_UNLOCK(scp);
	return (data);
}

static void
bcm_mi_writephy(struct bcm_softc *scp, u_int32_t reg, u_int32_t data)
{
	BCM_LOCK(scp);

	/* Clear MI interrupt */
	REG_WR(scp, emacintstatus, EI_MII);

	/* issue the write */
	REG_WR(scp, mdiodata, (MD_SB_START | MD_OP_WRITE |
		(scp->miphy << MD_PMD_SHIFT)
		| (reg << MD_RA_SHIFT) | MD_TA_VALID | (data & MD_DATA_MASK)));

	/* wait for it to complete */
	SPINWAIT(((REG_RD(scp, emacintstatus) & EI_MII) == 0), 100);
	if ((REG_RD(scp, emacintstatus) & EI_MII) == 0) {
		printf("bcm%d: PHY write timed out\n", scp->unit);
	}

	BCM_UNLOCK(scp);
}
  
static int
bcm_miibus_readreg(device_t device, int phy, int reg)
{
	struct bcm_softc 	*scp	= device_get_softc(device);
	int			data	= 0;

	BCM_LOCK(scp);

	if (phy != scp->miphy) {
		BCM_UNLOCK(scp);
		return (0);
	}

	data = bcm_mi_readphy(scp, reg);

	BCM_UNLOCK(scp);
	return (data);
}

static int
bcm_miibus_writereg(device_t device, int phy, int reg, int data)
{
	struct bcm_softc 	*scp	= device_get_softc(device);
	BCM_LOCK(scp);

	if (phy != scp->miphy) {
		BCM_UNLOCK(scp);
		return (0);
	}

	bcm_mi_writephy(scp, reg, data);

	BCM_UNLOCK(scp);
	return (0);
}
  
static void
bcm_miibus_statchg(device_t device)
{
	struct bcm_softc 	*scp	= device_get_softc(device);
	BCM_LOCK(scp);
	/* XXX what do we do here? */
	BCM_UNLOCK(scp);
	return;
}

/*
 *
 * Chip reset, boot, halt and interrupt functions
 *
 */

static void
bcm_chip_reset(struct bcm_softc *scp)
{
    	u_int32_t	bar0;

	BCM_LOCK(scp);

	bar0 = pci_read_config(scp->device, BCM_PCI_BAR0, 4);

	/* Point to enet core and get info */
	pci_write_config(scp->device, BCM_PCI_BAR0, BCM_SBBASE_ENET0, 4);
	bcm_chip_get_core(scp, &scp->enet_core);

	/* Point to pci core and get info */
	pci_write_config(scp->device, BCM_PCI_BAR0, BCM_SBBASE_PCI0, 4);
	bcm_chip_get_core(scp, &scp->pci_core);

	/* Restore original core */
	pci_write_config(scp->device, BCM_PCI_BAR0, bar0, 4);

	/* Reset core and phy */
	bcm_chip_reset_core(scp);
	bcm_chip_reset_phy(scp);

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_chip_reset_core(struct bcm_softc *scp)
{
    	u_int32_t	bar0, dummy;

	BCM_LOCK(scp);

	bar0 = pci_read_config(scp->device, BCM_PCI_BAR0, 4);

	/* Point to pci core and set interrupt vector for enet core */
	pci_write_config(scp->device, BCM_PCI_BAR0, BCM_SBBASE_PCI0, 4);
	REG_OR(scp, sbconfig.sbintvec, (scp->enet_core.instance ==  0) ? SBIV_ENET0 : SBIV_ENET1);

	dummy = REG_RD_OFFSET(scp, offsetof(sbpciregs_t, sbtopci2));
	dummy |= (SBTOPCI_PREF | SBTOPCI_BURST);
	REG_WR_OFFSET(scp, offsetof(sbpciregs_t, sbtopci2), dummy);

	/* Restore original core */
	pci_write_config(scp->device, BCM_PCI_BAR0, bar0, 4);

	/*
	 * If core already running then shut down the MAC and DMA.
	 */
	dummy = REG_RD(scp, sbconfig.sbtmstatelow) &
	    (SBTML_RESET | SBTML_REJ | SBTML_CLK);
	if (dummy == SBTML_CLK) {
BCM_XXX_PRINTF("bcm_chip_reset_core: core already running - shutting down\n");

		REG_WR(scp, intrecvlazy, 0);

		/* disable emac */
		REG_WR(scp, enetcontrol, EC_ED);
		SPINWAIT((REG_RD(scp, enetcontrol) & EC_ED), 200);

		/* reset the dma engines */
		REG_WR(scp, dmaregs.xmtcontrol, 0);

		if (REG_RD(scp, dmaregs.rcvstatus) & RS_RE_MASK) {
			/* wait until channel is idle or stopped */
			SPINWAIT(!(REG_RD(scp, dmaregs.rcvstatus) & RS_RS_IDLE), 100);
		}
		REG_WR(scp, dmaregs.rcvcontrol, 0);

		REG_WR(scp, enetcontrol, EC_ES);
		SPINWAIT((REG_RD(scp, enetcontrol) & EC_ES), 200);
	}

	/*
	 * Disable the core but not if already in reset
	 */
	dummy = REG_RD(scp, sbconfig.sbtmstatelow) & SBTML_RESET;
if (dummy)
	BCM_XXX_PRINTF("bcm_chip_reset_core: core already in reset state\n");
	if (!dummy) {
BCM_XXX_PRINTF("bcm_chip_reset_core: resetting core\n");
		
		/* set the reject bit */
		REG_WR(scp, sbconfig.sbtmstatelow, (SBTML_CLK | SBTML_REJ));

		/* spin until reject is set */
		while ((REG_RD(scp, sbconfig.sbtmstatelow) & SBTML_REJ) == 0)
			DELAY(1);

		/* spin until sbtmstatehigh.busy is clear */
		while (REG_RD(scp, sbconfig.sbtmstatehigh) & SBTMH_BUSY)
			DELAY(1);

		/* set reset and reject while enabling the clocks */
		REG_WR(scp, sbconfig.sbtmstatelow,
			(SBTML_FGC | SBTML_CLK | SBTML_REJ | SBTML_RESET));
		dummy = REG_RD(scp, sbconfig.sbtmstatelow);
		DELAY(10);

		/* leave reset and reject asserted */
		REG_WR(scp, sbconfig.sbtmstatelow, (SBTML_REJ | SBTML_RESET));
		dummy = REG_RD(scp, sbconfig.sbtmstatelow);
		DELAY(10);
	}

	/*
	 * Now do the initialization sequence.
	 */

	/* set reset while enabling the clock and forcing them on throughout the core */
	REG_WR(scp, sbconfig.sbtmstatelow,
	    (SBTML_FGC | SBTML_CLK | SBTML_RESET));
	dummy = REG_RD(scp, sbconfig.sbtmstatelow);
	DELAY(10);

	/* PR3158 workaround - not fixed in any chip yet */
	if (REG_RD(scp, sbconfig.sbtmstatehigh) & SBTMH_SERR) {
		printf("bcm%d: SBTMH_SERR; clearing...\n", scp->unit);
		REG_WR(scp, sbconfig.sbtmstatehigh, 0);
	}
	if (REG_RD(scp, sbconfig.sbimstate) & (SBIM_IBE | SBIM_TO)) {
		printf("bcm%d: SBIM_IBE | SBIM_TO; clearing...\n", scp->unit);
		REG_AND(scp, sbconfig.sbimstate, ~(SBIM_IBE | SBIM_TO));
	}

	/* clear reset and allow it to propagate throughout the core */
	REG_WR(scp, sbconfig.sbtmstatelow, (SBTML_FGC | SBTML_CLK));
	dummy = REG_RD(scp, sbconfig.sbtmstatelow);
	DELAY(10);
	
	/* leave clock enabled */
	REG_WR(scp, sbconfig.sbtmstatelow, SBTML_CLK);
	dummy = REG_RD(scp, sbconfig.sbtmstatelow);
	DELAY(10);

	/* Clear counters */
	bcm_chip_clear_stats(scp);

	/*
	 * We want the phy registers to be accessible even when
	 * the driver is "downed" so initialize MDC preamble, frequency,
	 * and whether internal or external phy here.
	 */

	/* 4402 has 62.5Mhz SB clock and internal phy */
	REG_WR(scp, mdiocontrol, 0x8d);

	/* some chips have internal phy, some don't */
	if (!(REG_RD(scp, devcontrol) & DC_IP)) {
		REG_WR(scp, enetcontrol, EC_EP);
	} else if (REG_RD(scp, devcontrol) & DC_ER) {
		REG_AND(scp, devcontrol, ~DC_ER);
		DELAY(100);
	}

	/* enable crc32 generation */
	REG_OR(scp, emaccontrol, EMC_CG);

	/* lazy RX interrupts */
	REG_WR(scp, intrecvlazy, ((BCM_XXX_LAZYRXFC << IRL_FC_SHIFT) & IRL_FC_MASK));
	if (BCM_XXX_LAZYRXFC > 1) {
	    /*
	     * XXX I think that this is wrong as it is writing into
	     * XXX the timeout field of the register. Unless it
	     * XXX actually meant to define the timeout and the
	     * XXX lazyrxmult is a bad name.
	     */
	    REG_OR(scp, intrecvlazy, ((BCM_XXX_LAZYRXMULT * BCM_XXX_LAZYRXFC) & IRL_TO_MASK));
	}

	/* set max frame lengths - account for possible vlan tag */
	REG_WR(scp, rxmaxlength, BCM_BUFFER_SIZE);
	REG_WR(scp, txmaxlength, BCM_BUFFER_SIZE);

	/* set tx watermark */
	REG_WR(scp, txwatermark, 56);

	/* initialize the tx and rx dma channels */
	REG_WR(scp, dmaregs.xmtcontrol, XC_XE);
	REG_WR(scp, dmaregs.xmtaddr, (scp->paddr_tx + BCM_SBBASE_PCIDMA));

	REG_WR(scp, dmaregs.rcvcontrol, (scp->rxoffset << RC_RO_SHIFT) | RC_RE);
	REG_WR(scp, dmaregs.rcvaddr, (scp->paddr_rx + BCM_SBBASE_PCIDMA));

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_chip_reset_phy(struct bcm_softc *scp)
{
    	int 			data;

	BCM_LOCK(scp);

	if (scp->miport != scp->enet_core.instance) {
		printf("bcm_chip_reset_phy: miport and core mismatch\n");
		BCM_UNLOCK(scp);
		return;
	}
#if BCM_XXX_PHYRESET_ONCE
	if (scp->phy_reset) {
		BCM_UNLOCK(scp);
		return;
	}
	scp->phy_reset = 1;
#endif /* BCM_XXX_PHYRESET_ONCE */

	bcm_mi_writephy(scp, 0, PHY_CTRL_PHY_RESET);
	DELAY(100);
	data = bcm_mi_readphy(scp, 0);
	if (data & PHY_CTRL_PHY_RESET) {
		printf("bcm%d: PHY reset not complete\n", scp->unit);
	}

	BCM_UNLOCK(scp);
	return;
}

/*
 * Enables the MAC and PHY. Requries that DMA buffers have been
 * set up.
 */
static void
bcm_chip_enable(struct bcm_softc *scp)
{
	BCM_LOCK(scp);

	bcm_chip_enable_phy(scp);
	bcm_chip_enable_core(scp);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Assumes that the TX and RX descriptor rings are set up.
 * Tells the chip where the DMA desriptors are.
 * Enables device interrupts and start the device.
 */
static void
bcm_chip_enable_core(struct bcm_softc *scp)
{
	BCM_LOCK(scp);

	/* turn on the emac */
	REG_OR(scp, enetcontrol, EC_EE);

	/* enable interrupts */
	REG_WR(scp, intmask, DEF_INTMASK);

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_chip_enable_phy(struct bcm_softc *scp)
{
	u_int32_t		Value32;

	BCM_LOCK(scp);

	/* enable activity led */
	Value32 = bcm_mi_readphy(scp, 26);
	bcm_mi_writephy(scp, 26, Value32 & 0x7fff);

	/* enable traffic meter led mode */
	Value32 = bcm_mi_readphy(scp, 27);
	bcm_mi_writephy(scp, 27, Value32 | (1 << 6));

#if BCM_XXX_EMAC_FLOWCONTROL
	/* Turnoff flow control */
	bcm_chip_flow(scp, 0, 0);
#endif /* BCM_XXX_EMAC_FLOWCONTROL */

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_chip_halt(struct bcm_softc *scp)
{
	BCM_LOCK(scp);

	bcm_chip_halt_core(scp);
	bcm_chip_halt_phy(scp);

	BCM_UNLOCK(scp);
	return;
}

static void
bcm_chip_halt_core(struct bcm_softc *scp)
{
	BCM_LOCK(scp);

	/* disable interrupts */
	REG_WR(scp, intmask, 0);
	(void)REG_RD(scp, intmask);	/* sync readback */

	/* disable emac */
	REG_WR(scp, enetcontrol, EC_ED);
	SPINWAIT((REG_RD(scp, enetcontrol) & EC_ED), 200);

	REG_WR(scp, dmaregs.xmtcontrol, 0);
	REG_WR(scp, dmaregs.rcvcontrol, 0);
	DELAY(10);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Isolate/power down the PHY, but leave the media selection
 * unchanged so that things will be put back to normal when
 * we bring the interface back up.
 */
static void
bcm_chip_halt_phy(struct bcm_softc *scp)
{
#if !BCM_XXX_PHYRESET_ONCE
	struct ifnet		*ifp;
	struct ifmedia_entry	*ifm;
	struct mii_data		*mii;
    	int 			itmp, mtmp;

	BCM_LOCK(scp);
 
	ifp = &scp->arpcom.ac_if;
	mii = device_get_softc(scp->miibus);

	itmp = ifp->if_flags;
	ifp->if_flags |= IFF_UP;
	ifm = mii->mii_media.ifm_cur;
	mtmp = ifm->ifm_media;
	ifm->ifm_media = IFM_ETHER|IFM_NONE;
	mii_mediachg(mii);
	ifm->ifm_media = mtmp;
	ifp->if_flags = itmp;

	BCM_UNLOCK(scp);
	return;
#endif /* !BCM_XXX_PHYRESET_ONCE */
}

/*
 * Get the instance and base address of a core
 */
static void
bcm_chip_get_core(struct bcm_softc *scp, struct bcm_core *core)
{
    	u_int32_t		match, sbid;

	BCM_LOCK(scp);


	sbid = REG_RD(scp, sbconfig.sbidhigh);
	core->core = (sbid & SBIDH_CC_MASK) >> SBIDH_CC_SHIFT;
	core->revision = (sbid & SBIDH_RC_MASK);
	core->vendor = (sbid & SBIDH_VC_MASK) >> SBIDH_VC_SHIFT;

	match = REG_RD(scp, sbconfig.sbadmatch0);
	core->type = match & SBAM_TYPE_MASK;

	if (core->type == 0)
		core->base = match & SBAM_BASE0_MASK;
	else
		/*
		 * XXX The Linux driver has code to cope with other cores and
		 * XXX address types. My chip is only type0 so we'll panic
		 * XXX and write the code later...
		 */
		panic("bcm%d: Strange type for core 0x%0x", scp->unit, match);

	switch (core->base) {
	case BCM_SBBASE_PCI0:
		core->instance = 0;
		break;
	case BCM_SBBASE_ENET0:
	    	core->instance = 0;
		break;
	default:
		panic("bcm%d: Unknown base for core 0x%0x", scp->unit, match);
	}

	if (bootverbose) {
	    printf("bcm%d: core 0x%x ", scp->unit, core->core);
	    printf("revi 0x%x ", core->revision);
	    printf("vend 0x%x ", core->vendor);
	    printf("match 0x%x (", match);
	    printf("type 0x%x ", core->type);
	    printf("base 0x%x) ", core->base);
	    printf("inst 0x%x\n", core->instance);
	}

	BCM_UNLOCK(scp);
	return;
}

/*
 * Clear on chip MIB registers
 */
static void
bcm_chip_clear_stats(struct bcm_softc *scp)
{
    	BCM_LOCK(scp);

	/* must clear mib registers by hand */
	REG_WR(scp, mibcontrol, EMC_RZ);
	(void) REG_RD(scp, mib.tx_good_octets);
	(void) REG_RD(scp, mib.tx_good_pkts);
	(void) REG_RD(scp, mib.tx_octets);
	(void) REG_RD(scp, mib.tx_pkts);
	(void) REG_RD(scp, mib.tx_broadcast_pkts);
	(void) REG_RD(scp, mib.tx_multicast_pkts);
	(void) REG_RD(scp, mib.tx_len_64);
	(void) REG_RD(scp, mib.tx_len_65_to_127);
	(void) REG_RD(scp, mib.tx_len_128_to_255);
	(void) REG_RD(scp, mib.tx_len_256_to_511);
	(void) REG_RD(scp, mib.tx_len_512_to_1023);
	(void) REG_RD(scp, mib.tx_len_1024_to_max);
	(void) REG_RD(scp, mib.tx_jabber_pkts);
	(void) REG_RD(scp, mib.tx_oversize_pkts);
	(void) REG_RD(scp, mib.tx_fragment_pkts);
	(void) REG_RD(scp, mib.tx_underruns);
	(void) REG_RD(scp, mib.tx_total_cols);
	(void) REG_RD(scp, mib.tx_single_cols);
	(void) REG_RD(scp, mib.tx_multiple_cols);
	(void) REG_RD(scp, mib.tx_excessive_cols);
	(void) REG_RD(scp, mib.tx_late_cols);
	(void) REG_RD(scp, mib.tx_defered);
	(void) REG_RD(scp, mib.tx_carrier_lost);
	(void) REG_RD(scp, mib.tx_pause_pkts);
	(void) REG_RD(scp, mib.rx_good_octets);
	(void) REG_RD(scp, mib.rx_good_pkts);
	(void) REG_RD(scp, mib.rx_octets);
	(void) REG_RD(scp, mib.rx_pkts);
	(void) REG_RD(scp, mib.rx_broadcast_pkts);
	(void) REG_RD(scp, mib.rx_multicast_pkts);
	(void) REG_RD(scp, mib.rx_len_64);
	(void) REG_RD(scp, mib.rx_len_65_to_127);
	(void) REG_RD(scp, mib.rx_len_128_to_255);
	(void) REG_RD(scp, mib.rx_len_256_to_511);
	(void) REG_RD(scp, mib.rx_len_512_to_1023);
	(void) REG_RD(scp, mib.rx_len_1024_to_max);
	(void) REG_RD(scp, mib.rx_jabber_pkts);
	(void) REG_RD(scp, mib.rx_oversize_pkts);
	(void) REG_RD(scp, mib.rx_fragment_pkts);
	(void) REG_RD(scp, mib.rx_missed_pkts);
	(void) REG_RD(scp, mib.rx_crc_align_errs);
	(void) REG_RD(scp, mib.rx_undersize);
	(void) REG_RD(scp, mib.rx_crc_errs);
	(void) REG_RD(scp, mib.rx_align_errs);
	(void) REG_RD(scp, mib.rx_symbol_errs);
	(void) REG_RD(scp, mib.rx_pause_pkts);
	(void) REG_RD(scp, mib.rx_nonpause_pkts);
	REG_WR(scp, mibcontrol, 0);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Grab the stats out of the core
 */
static void
bcm_chip_get_stats(struct bcm_softc *scp, struct bcm_mib *mib)
{
	BCM_LOCK(scp);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Set the RX Filter
 */
static void
bcm_chip_rxfilter(struct bcm_softc *scp)
{
	struct ifnet		*ifp;
	struct ifmultiaddr	*ifma;
    	u_int32_t		flags = 0;
	u_int32_t		idx = 0;

    	BCM_LOCK(scp);
	ifp = &scp->arpcom.ac_if;

	flags = REG_RD(scp, rxconfig);
BCM_XXX_PRINTF("bcm_rxfilter: flags were 0x%x", flags);

	if (ifp->if_flags & IFF_PROMISC) {
		flags |= ERC_PE;
		goto rxfilter;
	} else {
		flags &= ~ERC_PE;
	}

	if (ifp->if_flags & IFF_BROADCAST) {
		flags &= ~ERC_DB;
	} else {
		flags |= ERC_DB;
	}
		
	/* I hope this actually clears the CAM */
	REG_WR(scp, camcontrol, 0);

	/* our local address */
	bcm_chip_writecam(scp, scp->arpcom.ac_enaddr, idx++);

#if __FreeBSD_version > 500000
	/* allmulti or a list of discrete multicast addresses */
	if (ifp->if_flags & IFF_ALLMULTI)
		flags |= ERC_AM;
	else {
		flags &= ~ERC_AM;
		TAILQ_FOREACH(ifma, &ifp->if_multiaddrs, ifma_link) {
			if (ifma->ifma_addr->sa_family != AF_LINK)
				continue;
			bcm_chip_writecam(scp, LLADDR((struct sockaddr_dl *)ifma->ifma_addr), idx++);
		}
	}
#endif

	/* enable cam */
	REG_OR(scp, camcontrol, CC_CE);

rxfilter:
BCM_XXX_PRINTF(", flags now 0x%x\n", flags);
	REG_WR(scp, rxconfig, flags);

	BCM_UNLOCK(scp);
	return;
}

/*
 * Program the RX MAC address content addressable memory
 */
static void
bcm_chip_writecam(struct bcm_softc *scp, u_int8_t *ea, u_int32_t camindex)
{
	u_int32_t		 w;

	BCM_LOCK(scp);

	w = ((u_int32_t)ea[2] << 24) | ((u_int32_t)ea[3] << 16) |
		((u_int32_t) ea[4] << 8) | ea[5];
	REG_WR(scp, camdatalo, w);
	w = CD_V | ((u_int32_t)ea[0] << 8) | ea[1];
	REG_WR(scp, camdatahi, w);
	REG_WR(scp, camcontrol, (((u_int32_t) camindex << CC_INDEX_SHIFT) |
		CC_WR));

	/* spin until done */
	SPINWAIT((REG_RD(scp, camcontrol) & CC_CB), 100);

	BCM_UNLOCK(scp);
	return;
}
