/*
 * Definitions in the first part of this file are local to the
 * FreeBSD driver. They are prefixed with BCM_. Some
 * replace definitions in the Linux driver, others are
 * soley used in this driver.
 */

/* Aide memoires */
#define XXX_SOMETODO	static
#define XXX_ALLTODO	static
/* XXX Endian-ness */
/* XXX CRC off load */

/*
 * Driver configuration - currently these are to turn on and off
 * features whilst in development.
 */

/*
 * Override ring sizes - not less than 5
 */
#if 1
#define BCM_TX_LIST_CNT		511
#define BCM_RX_LIST_CNT		511
#endif

/*
 * Check in the watchdog for missed interrupts - left over from
 * earlier broken interrupt setup - just in case.
 */
#define BCM_XXX_INTRCHECK	1

/*
 * The Linux driver goes to some pains to ensure that the PHY is only
 * reset once and doesnot isolate it. We're going to try isolating
 * it when the interface goes down.
 */
#define BCM_XXX_PHYRESET_ONCE	1

/*
 * There might be RX packet coallesing with the following.
 * However, I don't understand the Linux use of lazyrxmult as
 * it seems to be setting bits in the timeout field of the
 * intrecvlazy register.
 */
#define BCM_XXX_LAZYRXFC	1
#define BCM_XXX_LAZYRXMULT	0

/*
 * The Linux driver implements 802.3 flow control by checking
 * whether the ends of the link can do it. At present just
 * have code where things needs to happen but do not
 * bother writing it.
 *
 * See Table 28B-3 of 802.3ab-1999 spec. and b44lm.c:b44_LM_SetFlowControl
 */
#define BCM_XXX_EMAC_FLOWCONTROL	0

/*
 * Debugging help
 */
#if 0
#define BCM_XXX_PRINTF(fmt, args...) do {	\
	printf(fmt, ##args);			\
} while (0)
#else
#define BCM_XXX_PRINTF(fmt, args...) do {	\
	if (bootverbose)			\
		printf(fmt, ##args);		\
} while (0)
#endif
#ifndef BCM_XXX_PRINTF
#define BCM_XXX_PRINTF(fmt, args...)
#endif

#if 0
#define BCM_XXX_TXDUMP(d, l) do {					\
	u_int8_t *dump;						\
	int j;							\
	dump = (d);						\
	for (j = 0; j < (l); j++) {				\
		if ((j > 1) && (j % 8 == 0))			\
			printf("\n");				\
		if (j % 8 == 0)					\
			printf("\t\t");				\
		printf("%02x ", *(dump+j));			\
	}							\
	printf("\n");						\
} while (0)
#else
#define BCM_XXX_TXDUMP(d, l)
#endif
#if 1
#define BCM_XXX_RXDUMP(d, l) do {					\
	u_int8_t *dump;						\
	int j;							\
	dump = (d);						\
	for (j = 0; j < (l); j++) {				\
		if ((j > 1) && (j % 8 == 0))			\
			printf("\n");				\
		if (j % 8 == 0)					\
			printf("\t\t");				\
		printf("%02x ", *(dump+j));			\
	}							\
	printf("\n");						\
} while (0)
#else
#define BCM_XXX_RXDUMP(d, l)
#endif

/*
 *
 * Macros and definitions for the driver
 *
 */

/*
 * Device recognition
 */
struct bcm_type {
	u_int16_t		bcm_vid;
	u_int16_t		bcm_did;
	char			*bcm_name;
};

#define BCOM_VENDORID		0x14E4
#define BCOM_DEVICEID_4401	0x4401

/*
 * Info per instance of the chips various cores.
 */
struct bcm_core {
	u_int32_t		core;
	u_int32_t		revision;
	u_int32_t		vendor;
	u_int32_t		instance;
	u_int32_t		base;
	u_int32_t		type;
};

/*
 * DMA Descriptors
 */
struct bcm_dma_desc {
	u_int32_t	ctrl;		/* misc control bits & bufcount */
	u_int32_t	addr;		/* data buffer address */
};

/*
 * Additional information for each DMA descriptor
 */
struct bcm_dmadd_bis {
	struct mbuf		*mbuf;
	bus_dmamap_t		dmamap;
};

/*
 * One of these per allocated device
 */
struct bcm_softc {
	struct arpcom		arpcom;

	/* PCI bus_space Stuff */
	bus_space_tag_t		bt;
	bus_space_handle_t	bh;
	vm_offset_t		vh;
	int			rid_memory;
	int			rid_irq;
	struct resource*	res_memory;
	struct resource*	res_irq;
	void			*intrhand;

	/* PCI driver stuff */
	struct bcm_core		enet_core;
	struct bcm_core		pci_core;

	/* Driver, miibus  and ethernet */
	device_t		device;
	int			unit;
#if __FreeBSD_version > 500000
	struct mtx		mtx;
#endif
	device_t		miibus;
	struct callout_handle	stat_ch;

	/* PHY */
	int			miphy;
	int			miport;
	int			phy_reset;

	/* DMA Details */
	bus_dma_tag_t		tag_parent;
	bus_dma_tag_t		tag_rx;
	bus_dma_tag_t		tag_tx;
	bus_dma_tag_t		tag_mbuf;
	bus_dmamap_t		dmamap_rx;
	bus_dmamap_t		dmamap_tx;
	u_int32_t		paddr_rx;
	u_int32_t		paddr_tx;
	struct bcm_dma_desc	*desc_rx;	/* descriptor list */
	struct bcm_dma_desc	*desc_tx;	/* descriptor list */
	struct bcm_dmadd_bis	desc_rx_bis[BCM_RX_LIST_CNT];/* driver info for each desc */
	struct bcm_dmadd_bis	desc_tx_bis[BCM_TX_LIST_CNT];/* driver info for each desc */
	int			con_rx;		/* next descriptor to free */
	int			idx_tx;		/* next free descriptor */
	int			cnt_tx;		/* number descriptors in use*/
	int			con_tx;		/* next descriptor to free */
	int			dma_rx;		/* descriptor for DMA */
	int			dma_tx;		/* descriptor for DMA */
	int			rxoffset;	/* rx offset for header */
};

/*
 * Locking
 */
#if __FreeBSD_version > 500000
#if 1
#define	BCM_LOCK(scp)		mtx_lock(&scp->mtx)
#define	BCM_UNLOCK(scp)		mtx_unlock(&scp->mtx)
#else
#define	BCM_LOCK(scp)		do {				\
	mtx_lock(&scp->mtx);					\
	printf("%s:%d: entry\n", __func__, scp->XXX_depth++);	\
} while (0)
#define	BCM_UNLOCK(scp)		do {				\
	printf("%s:%d: exit\n", __func__, --scp->XXX_depth);	\
	mtx_unlock(&scp->mtx);					\
} while (0)
#endif
#else
#define BCM_LOCK(scp)       int s;s=splimp()
#define BCM_UNLOCK(scp)     splx(s)
#endif

/*
 * Start of EPROM in memory map.
 */
#define BCM_MEM_EPROM		0x1000

/*
 * Addresses of various cores for the windowed bus
 */
#define BCM_SBBASE_PCIDMA	0x40000000
#define BCM_SBBASE_ENET0	0x18000000
#define BCM_SBBASE_PCI0		0x18002000

/*
 * PCI Registers
 */
#define BCM_PCI_VID			0x00
#define BCM_PCI_DID			0x02
#define BCM_PCI_REV			0x08
#define BCM_PCI_MEMLO			0x10
#define BCM_PCI_MEMHIGH			0x14
#define BCM_PCI_INTLINE			0x3C
#define BCM_PCI_BAR0			0x80

/*
 * Buffers and rings
 */

/* Receive header rounded to 30bytes total */
struct bcm_rxh {
	u_int16_t	len;
	u_int16_t	flags;
	u_int16_t	pad[13];
};

/* Maximum buffer size */
#define BCM_BUFFER_SIZE		(ETHER_MAX_LEN+32)

/* Ring lengths */
#ifndef BCM_TX_LIST_CNT		
#define BCM_TX_LIST_CNT		(DMAMAXRINGSZ/sizeof(struct bcm_dma_desc))
#endif /* BCM_TX_LIST_CNT */
#ifndef BCM_RX_LIST_CNT		
#define BCM_RX_LIST_CNT		(DMAMAXRINGSZ/sizeof(struct bcm_dma_desc))
#endif /* BCM_RX_LIST_CNT */
#define BCM_INC(x, y)		(x) = ((x) == ((y)-1)) ? 0 : (x)+1

/*
 * FreeBSD Version of Linux macros
 */
#define REG_RD(scp, reg) \
	bus_space_read_4(scp->bt, scp->bh, offsetof(bcmenetregs_t, reg))

#define REG_RD_OFFSET(scp, reg) \
	bus_space_read_4(scp->bt, scp->bh, reg)

#define REG_WR(scp, reg, val) \
	bus_space_write_4(scp->bt, scp->bh, offsetof(bcmenetregs_t, reg) , (val))

#define REG_WR_OFFSET(scp, reg, val) \
	bus_space_write_4(scp->bt, scp->bh, reg, (val))

#define REG_OR(scp, reg, val) do {				\
	volatile u_int32_t v;					\
	v = REG_RD(scp, reg);					\
	v |= val;						\
	REG_WR(scp, reg, v);					\
} while (0)

#define REG_AND(scp, reg, val) do {				\
	volatile u_int32_t v;					\
	v = REG_RD(scp, reg);					\
	v &= val;						\
	REG_WR(scp, reg, v);					\
} while (0)

#define SPINWAIT(exp, us) { \
	u_int32_t countdown = (us) + 9; \
	while ((exp) && (countdown >= 10)) {\
		DELAY(10); \
		countdown -= 10; \
	} \
}

/*
 *
 * From here on are definitions taken verbatim from the
 * Linux driver
 *
 */

/* For importing lines from the Linux driver */
#define LM_UINT16	u_int16_t
#define LM_UINT32	u_int32_t


/******************************************************************************/
/* MII registers. */
/******************************************************************************/

#define BIT_15				    0x8000

#define PHY_CTRL_REG                                0x00
#define PHY_CTRL_PHY_RESET			    BIT_15

/******************************************************************************/
/* Register definitions. */
/******************************************************************************/

/*
 * Each DMA processor consists of a transmit channel and a receive channel.
 */
typedef volatile struct {
	/* transmit channel */
	LM_UINT32	xmtcontrol;			/* enable, et al */
	LM_UINT32	xmtaddr;			/* descriptor ring base address (4K aligned) */
	LM_UINT32	xmtptr;				/* last descriptor posted to chip */
	LM_UINT32	xmtstatus;			/* current active descriptor, et al */

	/* receive channel */
	LM_UINT32	rcvcontrol;			/* enable, et al */
	LM_UINT32	rcvaddr;			/* descriptor ring base address (4K aligned) */
	LM_UINT32	rcvptr;				/* last descriptor posted to chip */
	LM_UINT32	rcvstatus;			/* current active descriptor, et al */

	/* diag access */
	LM_UINT32	fifoaddr;			/* diag address */
	LM_UINT32	fifodatalow;			/* low 32bits of data */
	LM_UINT32	fifodatahigh;			/* high 32bits of data */
	LM_UINT32	pad;				/* reserved */
} dmaregs_t;

/* transmit channel control */
#define	XC_XE		((LM_UINT32)1 << 0)	/* transmit enable */
#define	XC_SE		((LM_UINT32)1 << 1)	/* transmit suspend request */
#define	XC_LE		((LM_UINT32)1 << 2)	/* loopback enable */
#define	XC_FPRI		((LM_UINT32)1 << 3)	/* fair priority */
#define	XC_FL		((LM_UINT32)1 << 4)	/* flush request */

/* transmit descriptor table pointer */
#define	XP_LD_MASK	0xfff			/* last valid descriptor */

/* transmit channel status */
#define	XS_CD_MASK	0x0fff			/* current descriptor pointer */
#define	XS_XS_MASK	0xf000			/* transmit state */
#define	XS_XS_SHIFT	12
#define	XS_XS_DISABLED	0x0000			/* disabled */
#define	XS_XS_ACTIVE	0x1000			/* active */
#define	XS_XS_IDLE	0x2000			/* idle wait */
#define	XS_XS_STOPPED	0x3000			/* stopped */
#define	XS_XS_SUSP	0x4000			/* suspend pending */
#define	XS_XE_MASK	0xf0000			/* transmit errors */
#define	XS_XE_SHIFT	16
#define	XS_XE_NOERR	0x00000			/* no error */
#define	XS_XE_DPE	0x10000			/* descriptor protocol error */
#define	XS_XE_DFU	0x20000			/* data fifo underrun */
#define	XS_XE_BEBR	0x30000			/* bus error on buffer read */
#define	XS_XE_BEDA	0x40000			/* bus error on descriptor access */
#define	XS_FL		((LM_UINT32)1 << 20)	/* flushed */

/* receive channel control */
#define	RC_RE		((LM_UINT32)1 << 0)	/* receive enable */
#define	RC_RO_MASK	0xfe			/* receive frame offset */
#define	RC_RO_SHIFT	1

/* receive descriptor table pointer */
#define	RP_LD_MASK	0xfff			/* last valid descriptor */

/* receive channel status */
#define	RS_CD_MASK	0x0fff			/* current descriptor pointer */
#define	RS_RS_MASK	0xf000			/* receive state */
#define	RS_RS_SHIFT	12
#define	RS_RS_DISABLED	0x0000			/* disabled */
#define	RS_RS_ACTIVE	0x1000			/* active */
#define	RS_RS_IDLE	0x2000			/* idle wait */
#define	RS_RS_STOPPED	0x3000			/* reserved */
#define	RS_RE_MASK	0xf0000			/* receive errors */
#define	RS_RE_SHIFT	16
#define	RS_RE_NOERR	0x00000			/* no error */
#define	RS_RE_DPE	0x10000			/* descriptor protocol error */
#define	RS_RE_DFO	0x20000			/* data fifo overflow */
#define	RS_RE_BEBW	0x30000			/* bus error on buffer write */
#define	RS_RE_BEDA	0x40000			/* bus error on descriptor access */

/* fifoaddr */
#define	FA_OFF_MASK	0xffff			/* offset */
#define	FA_SEL_MASK	0xf0000			/* select */
#define	FA_SEL_SHIFT	16
#define	FA_SEL_XDD	0x00000			/* transmit dma data */
#define	FA_SEL_XDP	0x10000			/* transmit dma pointers */
#define	FA_SEL_RDD	0x40000			/* receive dma data */
#define	FA_SEL_RDP	0x50000			/* receive dma pointers */
#define	FA_SEL_XFD	0x80000			/* transmit fifo data */
#define	FA_SEL_XFP	0x90000			/* transmit fifo pointers */
#define	FA_SEL_RFD	0xc0000			/* receive fifo data */
#define	FA_SEL_RFP	0xd0000			/* receive fifo pointers */

/*
 * Each descriptor ring must be 4096byte aligned
 * and fit within a single 4096byte page.
 */
#define	DMAMAXRINGSZ	4096
#define	DMARINGALIGN	4096

/* control flags */
#define	CTRL_BC_MASK	0x1fff			/* buffer byte count */
#define	CTRL_EOT	((LM_UINT32)1 << 28)	/* end of descriptor table */
#define	CTRL_IOC	((LM_UINT32)1 << 29)	/* interrupt on completion */
#define	CTRL_EOF	((LM_UINT32)1 << 30)	/* end of frame */
#define	CTRL_SOF	((LM_UINT32)1 << 31)	/* start of frame */

/* control flags in the range [27:20] are core-specific and not defined here */
#define	CTRL_CORE_MASK	0x0ff00000

/* cpp contortions to concatenate w/arg prescan */
#ifndef PAD
#define	_PADLINE(line)	pad ## line
#define	_XSTR(line)	_PADLINE(line)
#define	PAD		_XSTR(__LINE__)
#endif	/* PAD */

/*
 * EMAC MIB Registers
 */
typedef volatile struct {
	LM_UINT32 tx_good_octets;
	LM_UINT32 tx_good_pkts;
	LM_UINT32 tx_octets;
	LM_UINT32 tx_pkts;
	LM_UINT32 tx_broadcast_pkts;
	LM_UINT32 tx_multicast_pkts;
	LM_UINT32 tx_len_64;
	LM_UINT32 tx_len_65_to_127;
	LM_UINT32 tx_len_128_to_255;
	LM_UINT32 tx_len_256_to_511;
	LM_UINT32 tx_len_512_to_1023;
	LM_UINT32 tx_len_1024_to_max;
	LM_UINT32 tx_jabber_pkts;
	LM_UINT32 tx_oversize_pkts;
	LM_UINT32 tx_fragment_pkts;
	LM_UINT32 tx_underruns;
	LM_UINT32 tx_total_cols;
	LM_UINT32 tx_single_cols;
	LM_UINT32 tx_multiple_cols;
	LM_UINT32 tx_excessive_cols;
	LM_UINT32 tx_late_cols;
	LM_UINT32 tx_defered;
	LM_UINT32 tx_carrier_lost;
	LM_UINT32 tx_pause_pkts;
	LM_UINT32 PAD[8];

	LM_UINT32 rx_good_octets;
	LM_UINT32 rx_good_pkts;
	LM_UINT32 rx_octets;
	LM_UINT32 rx_pkts;
	LM_UINT32 rx_broadcast_pkts;
	LM_UINT32 rx_multicast_pkts;
	LM_UINT32 rx_len_64;
	LM_UINT32 rx_len_65_to_127;
	LM_UINT32 rx_len_128_to_255;
	LM_UINT32 rx_len_256_to_511;
	LM_UINT32 rx_len_512_to_1023;
	LM_UINT32 rx_len_1024_to_max;
	LM_UINT32 rx_jabber_pkts;
	LM_UINT32 rx_oversize_pkts;
	LM_UINT32 rx_fragment_pkts;
	LM_UINT32 rx_missed_pkts;
	LM_UINT32 rx_crc_align_errs;
	LM_UINT32 rx_undersize;
	LM_UINT32 rx_crc_errs;
	LM_UINT32 rx_align_errs;
	LM_UINT32 rx_symbol_errs;
	LM_UINT32 rx_pause_pkts;
	LM_UINT32 rx_nonpause_pkts;
} bcmenetmib_t;

/*
 * Sonics Configuration Space Registers.
 */
typedef volatile struct _sbconfig {
	LM_UINT32	PAD[2];
	LM_UINT32	sbipsflag;		/* initiator port ocp slave flag */
	LM_UINT32	PAD[3];
	LM_UINT32	sbtpsflag;		/* target port ocp slave flag */
	LM_UINT32	PAD[17];
	LM_UINT32	sbadmatch3;		/* address match3 */
	LM_UINT32	PAD;
	LM_UINT32	sbadmatch2;		/* address match2 */
	LM_UINT32	PAD;
	LM_UINT32	sbadmatch1;		/* address match1 */
	LM_UINT32	PAD[7];
	LM_UINT32	sbimstate;		/* initiator agent state */
	LM_UINT32	sbintvec;		/* interrupt mask */
	LM_UINT32	sbtmstatelow;		/* target state */
	LM_UINT32	sbtmstatehigh;		/* target state */
	LM_UINT32	sbbwa0;			/* bandwidth allocation table0 */
	LM_UINT32	PAD;
	LM_UINT32	sbimconfiglow;		/* initiator configuration */
	LM_UINT32	sbimconfighigh;		/* initiator configuration */
	LM_UINT32	sbadmatch0;		/* address match0 */
	LM_UINT32	PAD;
	LM_UINT32	sbtmconfiglow;		/* target configuration */
	LM_UINT32	sbtmconfighigh;		/* target configuration */
	LM_UINT32	sbbconfig;		/* broadcast configuration */
	LM_UINT32	PAD;
	LM_UINT32	sbbstate;		/* broadcast state */
	LM_UINT32	PAD[3];
	LM_UINT32	sbactcnfg;		/* activate configuration */
	LM_UINT32	PAD[3];
	LM_UINT32	sbflagst;		/* current sbflags */
	LM_UINT32	PAD[3];
	LM_UINT32	sbidlow;		/* identification */
	LM_UINT32	sbidhigh;		/* identification */
} sbconfig_t;

/* sbimstate */
#define SBIMSTATE	0x90		/* offset */
#define	SBIM_PC		0xf		/* pipecount */
#define	SBIM_AP_MASK	0x30		/* arbitration policy */
#define	SBIM_AP_BOTH	0x00		/* use both timeslaces and token */
#define	SBIM_AP_TS	0x10		/* use timesliaces only */
#define	SBIM_AP_TK	0x20		/* use token only */
#define	SBIM_AP_RSV	0x30		/* reserved */
#define	SBIM_IBE	0x20000		/* inbanderror */
#define	SBIM_TO		0x40000		/* timeout */

/* sbadmatch0 */
#define SBADMATCH0		0xB0		/* offset */
#define	SBAM_TYPE_MASK		0x3		/* address type */
#define	SBAM_BASE0_MASK		0xffffff00	/* type0 base address */
#define	SBAM_BASE0_SHIFT	8
#define	SBAM_BASE1_MASK		0xfffff000	/* type1 base address for the core */
#define	SBAM_BASE1_SHIFT	12
#define	SBAM_BASE2_MASK		0xffff0000	/* type2 base address for the core */
#define	SBAM_BASE2_SHIFT	16

/* sbintvec */
#define SBINTVEC	0x94		/* offset */
#define	SBIV_PCI	0x1		/* enable interrupts for pci */
#define	SBIV_ENET0	0x2		/* enable interrupts for enet 0 */
#define	SBIV_ILINE20	0x4		/* enable interrupts for iline20 */
#define	SBIV_CODEC	0x8		/* enable interrupts for v90 codec */
#define	SBIV_USB	0x10		/* enable interrupts for usb */
#define	SBIV_EXTIF	0x20		/* enable interrupts for external i/f */
#define	SBIV_ENET1	0x40		/* enable interrupts for enet 1 */

/* sbtmstatelow */
#define SBTMSTATELOW	0x98		/* offset */
#define	SBTML_RESET	0x1		/* reset */
#define	SBTML_REJ	0x2		/* reject */
#define	SBTML_CLK	0x10000		/* clock enable */
#define	SBTML_FGC	0x20000		/* force gated clocks on */
#define	SBTML_PE	0x40000000	/* pme enable */
#define	SBTML_BE	0x80000000	/* bist enable */

/* sbtmstatehigh */
#define SBTMSTATEHIGH	0x9C		/* offset */
#define	SBTMH_SERR	0x1		/* serror */
#define	SBTMH_INT	0x2		/* interrupt */
#define	SBTMH_BUSY	0x4		/* busy */
#define	SBTMH_GCR	0x20000000	/* gated clock request */
#define	SBTMH_BISTF	0x40000000	/* bist failed */
#define	SBTMH_BISTD	0x80000000	/* bist done */

/* sbidhigh */
#define SBIDHIGH	0xFC		/* offset */
#define	SBIDH_RC_MASK	0xf		/* revision code*/
#define	SBIDH_CC_MASK	0xfff0		/* core code */
#define	SBIDH_CC_SHIFT	4
#define	SBIDH_VC_MASK	0xffff0000	/* vendor code */
#define	SBIDH_VC_SHIFT	16

/*
 * Host Interface Registers
 */
typedef volatile struct _bcmenettregs {
	/* Device and Power Control */
	LM_UINT32	devcontrol;
	LM_UINT32	PAD[2];
	LM_UINT32	biststatus;
	LM_UINT32	wakeuplength;
	LM_UINT32	PAD[3];
	
	/* Interrupt Control */
	LM_UINT32	intstatus;
	LM_UINT32	intmask;
	LM_UINT32	gptimer;
	LM_UINT32	PAD[23];

	/* Ethernet MAC Address Filtering Control */
	LM_UINT32	PAD[2];
	LM_UINT32	enetftaddr;
	LM_UINT32	enetftdata;
	LM_UINT32	PAD[2];

	/* Ethernet MAC Control */
	LM_UINT32	emactxmaxburstlen;
	LM_UINT32	emacrxmaxburstlen;
	LM_UINT32	emaccontrol;
	LM_UINT32	emacflowcontrol;

	LM_UINT32	PAD[20];

	/* DMA Lazy Interrupt Control */
	LM_UINT32	intrecvlazy;
	LM_UINT32	PAD[63];

	/* DMA engine */
	dmaregs_t	dmaregs;
	LM_UINT32	PAD[116];

	/* EMAC Registers */
	LM_UINT32 rxconfig;
	LM_UINT32 rxmaxlength;
	LM_UINT32 txmaxlength;
	LM_UINT32 PAD;
	LM_UINT32 mdiocontrol;
	LM_UINT32 mdiodata;
	LM_UINT32 emacintmask;
	LM_UINT32 emacintstatus;
	LM_UINT32 camdatalo;
	LM_UINT32 camdatahi;
	LM_UINT32 camcontrol;
	LM_UINT32 enetcontrol;
	LM_UINT32 txcontrol;
	LM_UINT32 txwatermark;
	LM_UINT32 mibcontrol;
	LM_UINT32 PAD[49];

	/* EMAC MIB counters */
	bcmenetmib_t	mib;

	LM_UINT32	PAD[585];

	/* Sonics SiliconBackplane config registers */
	sbconfig_t	sbconfig;
} bcmenetregs_t;

/* device control */
#define	DC_PM		((LM_UINT32)1 << 7)	/* pattern filtering enable */
#define	DC_IP		((LM_UINT32)1 << 10)	/* internal ephy present (rev >= 1) */
#define	DC_ER		((LM_UINT32)1 << 15)	/* ephy reset */
#define	DC_MP		((LM_UINT32)1 << 16)	/* mii phy mode enable */
#define	DC_CO		((LM_UINT32)1 << 17)	/* mii phy mode: enable clocks */
#define	DC_PA_MASK	0x7c0000		/* mii phy mode: mdc/mdio phy address */
#define	DC_PA_SHIFT	18

/* intstatus and intmask */
#define	I_PME		((LM_UINT32)1 << 6)	/* power management event */
#define	I_TO		((LM_UINT32)1 << 7)	/* general purpose timeout */
#define	I_PC		((LM_UINT32)1 << 10)	/* descriptor error */
#define	I_PD		((LM_UINT32)1 << 11)	/* data error */
#define	I_DE		((LM_UINT32)1 << 12)	/* descriptor protocol error */
#define	I_RU		((LM_UINT32)1 << 13)	/* receive descriptor underflow */
#define	I_RO		((LM_UINT32)1 << 14)	/* receive fifo overflow */
#define	I_XU		((LM_UINT32)1 << 15)	/* transmit fifo underflow */
#define	I_RI		((LM_UINT32)1 << 16)	/* receive interrupt */
#define	I_XI		((LM_UINT32)1 << 24)	/* transmit interrupt */
#define	I_EM		((LM_UINT32)1 << 26)	/* emac interrupt */
#define	I_MW		((LM_UINT32)1 << 27)	/* mii write */
#define	I_MR		((LM_UINT32)1 << 28)	/* mii read */

#define	I_ERRORS	(I_PC | I_PD | I_DE | I_RU | I_RO | I_XU)
#define	DEF_INTMASK	(I_XI | I_RI | I_ERRORS)

/* emaccontrol */
#define	EMC_CG		((LM_UINT32)1 << 0)	/* crc32 generation enable */
#define	EMC_EP		((LM_UINT32)1 << 2)	/* onchip ephy: powerdown (rev >= 1) */
#define	EMC_ED		((LM_UINT32)1 << 3)	/* onchip ephy: energy detected (rev >= 1) */
#define	EMC_LC_MASK	0xe0			/* onchip ephy: led control (rev >= 1) */
#define	EMC_LC_SHIFT	5

/* interrupt receive lazy */
#define	IRL_TO_MASK	0x00ffffff		/* timeout */
#define	IRL_FC_MASK	0xff000000		/* frame count */
#define	IRL_FC_SHIFT	24			/* frame count */

/* emac receive config */
#define	ERC_DB		((LM_UINT32)1 << 0)	/* disable broadcast */
#define	ERC_AM		((LM_UINT32)1 << 1)	/* accept all multicast */
#define	ERC_RDT		((LM_UINT32)1 << 2)	/* receive disable while transmitting */
#define	ERC_PE		((LM_UINT32)1 << 3)	/* promiscuous enable */
#define	ERC_LE		((LM_UINT32)1 << 4)	/* loopback enable */
#define	ERC_EF		((LM_UINT32)1 << 5)	/* enable flow control */
#define	ERC_UF		((LM_UINT32)1 << 6)	/* accept unicast flow control frame */
#define	ERC_RF		((LM_UINT32)1 << 7)	/* reject filter */

/* emac mdio data */
#define	MD_DATA_MASK	0xffffL			/* r/w data */
#define	MD_TA_MASK	0x30000L		/* turnaround value */
#define	MD_TA_SHIFT	16
#define	MD_TA_VALID	(2L << MD_TA_SHIFT)	/* valid ta */
#define	MD_RA_MASK	0x7c0000L		/* register address */
#define	MD_RA_SHIFT	18
#define	MD_PMD_MASK	0xf800000L		/* physical media device */
#define	MD_PMD_SHIFT	23
#define	MD_OP_MASK	0x30000000L		/* opcode */
#define	MD_OP_SHIFT	28
#define	MD_OP_WRITE	(1L << MD_OP_SHIFT)	/* write op */
#define	MD_OP_READ	(2L << MD_OP_SHIFT)	/* read op */
#define	MD_SB_MASK	0xc0000000L		/* start bits */
#define	MD_SB_SHIFT	30
#define	MD_SB_START	(0x1L << MD_SB_SHIFT)	/* start of frame */

/* emac intstatus and intmask */
#define	EI_MII		((LM_UINT32)1 << 0)	/* mii mdio interrupt */
#define	EI_MIB		((LM_UINT32)1 << 1)	/* mib interrupt */
#define	EI_FLOW		((LM_UINT32)1 << 2)	/* flow control interrupt */

/* emac cam data high */
#define	CD_V		((LM_UINT32)1 << 16)	/* valid bit */

/* emac cam control */
#define	CC_CE		((LM_UINT32)1 << 0)	/* cam enable */
#define	CC_MS		((LM_UINT32)1 << 1)	/* mask select */
#define	CC_RD		((LM_UINT32)1 << 2)	/* read */
#define	CC_WR		((LM_UINT32)1 << 3)	/* write */
#define	CC_INDEX_MASK	0x3f0000		/* index */
#define	CC_INDEX_SHIFT	16
#define	CC_CB		((LM_UINT32)1 << 31)	/* cam busy */

/* emac ethernet control */
#define	EC_EE		((LM_UINT32)1 << 0)	/* emac enable */
#define	EC_ED		((LM_UINT32)1 << 1)	/* emac disable */
#define	EC_ES		((LM_UINT32)1 << 2)	/* emac soft reset */
#define	EC_EP		((LM_UINT32)1 << 3)	/* external phy select */

/* emac mib control */
#define	EMC_RZ		((LM_UINT32)1 << 0)	/* autoclear on read */

/* rx header */
#define	RXF_L		((LM_UINT16)1 << 11)	/* last buffer in a frame */
#define	RXF_MISS	((LM_UINT16)1 << 7)	/* received due to promisc mode */
#define	RXF_BRDCAST	((LM_UINT16)1 << 6)	/* dest is broadcast address */
#define	RXF_MULT	((LM_UINT16)1 << 5)	/* dest is multicast address */
#define	RXF_LG		((LM_UINT16)1 << 4)	/* frame length > rxmaxlength */
#define	RXF_NO		((LM_UINT16)1 << 3)	/* odd number of nibbles */
#define	RXF_RXER	((LM_UINT16)1 << 2)	/* receive symbol error */
#define	RXF_CRC		((LM_UINT16)1 << 1)	/* crc error */
#define	RXF_OV		((LM_UINT16)1 << 0)	/* fifo overflow */

#define RXF_ERRORS (RXF_NO | RXF_RXER | RXF_CRC | RXF_OV)

/* Sonics side: PCI core and host control registers */
typedef struct sbpciregs {
	LM_UINT32 control;		/* PCI control */
	LM_UINT32 PAD[3];
	LM_UINT32 arbcontrol;	/* PCI arbiter control */
	LM_UINT32 PAD[3];
	LM_UINT32 intstatus;	/* Interrupt status */
	LM_UINT32 intmask;		/* Interrupt mask */
	LM_UINT32 sbtopcimailbox;	/* Sonics to PCI mailbox */
	LM_UINT32 PAD[9];
	LM_UINT32 bcastaddr;	/* Sonics broadcast address */
	LM_UINT32 bcastdata;	/* Sonics broadcast data */
	LM_UINT32 PAD[42];
	LM_UINT32 sbtopci0;	/* Sonics to PCI translation 0 */
	LM_UINT32 sbtopci1;	/* Sonics to PCI translation 1 */
	LM_UINT32 sbtopci2;	/* Sonics to PCI translation 2 */
	LM_UINT32 PAD[445];
	LM_UINT16 sprom[36];	/* SPROM shadow Area */
	LM_UINT32 PAD[46];
} sbpciregs_t;

/* Sonics to PCI translation types */
#define SBTOPCI0_MASK	0xfc000000
#define SBTOPCI1_MASK	0xfc000000
#define SBTOPCI2_MASK	0xc0000000
#define SBTOPCI_MEM	0
#define SBTOPCI_IO	1
#define SBTOPCI_CFG0	2
#define SBTOPCI_CFG1	3
#define	SBTOPCI_PREF	0x4	/* prefetch enable */
#define	SBTOPCI_BURST	0x8	/* burst enable */
